<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Libraries\{ Stripe, Paypal, Skrill, Razorpay, Iyzicolib, Coingate, Midtrans, Paymentwall, Authorizenet,
                    Paystack, Adyen, Instamojo, Offlinepayment, Payhere, Coinpayments, Spankpay, Omise, Sslcommerz, Flutterwave };
use App\Models\{ Transaction, Coupon, Product, Skrill_Transaction, Pricing_Table, User_Subscription, Key, Affiliate_Earning, 
                  Transaction_Note, User, Prepaid_Credit, User_Prepaid_Credit, User_Shopping_Cart_Item };
use Illuminate\Support\Facades\{ DB, Session, Cache, Validator, Auth };
use Ramsey\Uuid;


class CheckoutController extends Controller
{
    public $transaction_details = [];
    public $transaction_params = [];
    public $payment_link = null;
    public $pending_transaction;


    public function checkout_error(Request $request)
    {
        $message = session('message') ?? abort(404);

        config([
            "meta_data.title" => __('Payment failed'),
        ]);

        return view_('checkout_error', compact('message'));
    }


    public function payment(Request $request, $return_url = false, $user = null)
    { 
        $checkout_config = $request->post('checkout_config') ?? abort(404);
        $checkout_config = decrypt($checkout_config, false);
        $checkout_config = json_decode($checkout_config, true) ?? abort(404);

        extract($checkout_config);

        if(count($cart_items) && \Validator::make(['cart_id' => $request->post('cart_id')], ['cart_id' => 'uuid|required'])->fails())
        {
            return redirect()->back()->with(['user_message' => __('Something wrong happened')]);
        }

        if(!($payment_gateway['name'] ?? null))
        {
            return redirect()->back()->with(['user_message' => __('Please select a payment method')]);
        }

        $processor   = $payment_gateway['name'] ?? null;
        $tos         = filter_var($request->post('tos'), FILTER_VALIDATE_BOOLEAN);
        $user_notes  = filter_var($request->post('notes'), FILTER_SANITIZE_STRING);
        $guest_email = $request->post('guest_email');
        $guest_token = ($user ?? Auth::check()) ? null : uuid6();

        if(config('payments.tos') && !$request->post('tos'))
        {
            $message = ['user_message' => __('Please agree to our terms and conditions.')];

            if(config("payment_gateways.{$processor}.async"))
            {
                return json($message);
            }

            return back()->withInput()->with($message);
        }

        if(!\Auth::check() && !filter_var($guest_email, FILTER_VALIDATE_EMAIL))
        {
            $message = ['user_message' => __('Wrong or missing guest email address.')];

            if(config("payment_gateways.{$processor}.async"))
            {
                return json($message);
            }

            return back()->withInput()->with($message);
        }

        $invalid_user_notes = Validator::make($request->all(), ['notes' => "nullable|string|max:5000"])->fails();

        if(config('payments.buyer_note') && !$subscription && $invalid_user_notes)
        {
            $message = ['user_message' => __('There is something wrong with the notes field.')];

            if(config("payment_gateways.{$processor}.async"))
            {
                return json($message);
            }

            return back()->with($message);
        }

        $minimum_custom_amount = config("payments_gateways.{$processor}.minimum", 0);

        $validator =  Validator::make($request->all(), [
                          'custom_amount' => "nullable|numeric|digits_between:0,25|gte:{$minimum_custom_amount}",
                      ]);

        if($validator->fails())
        {
            return back()->with(['user_message' => __('The custom amount is insufficient to proceed the payment')]);
        }

        if(count($cart_items))
        {
            foreach($cart_items as &$item)
            {
                $item['license'] = null;

                if(Product::find($item['item_id'])->enable_license)
                {
                    $item['license'] = uuid6();
                }
            }
        }

        $details = [
            'items'           => $cart_items,
            'subscription'    => $subscription,
            'prepaid_credit'  => $prepaid_credit,
            'processor'       => $processor,
            'amount'          => $due_amount,
            'discount'        => $discount,
            'fee'             => $fee,
            'tax'             => $tax,
            'type'            => $type,
            'currency'        => null,
            'exchange_rate'   => null,
            'reference_id'    => null,
        ];

        $transaction = [
            'reference_id'    => $gateway->details['reference'] ?? null,
            'user_id'         => Auth::id(),
            'processor'       => null,
            'details'         => null,
            'amount'          => $due_amount,
            'custom_amount'   => $due_amount,
            'discount'        => $discount,
            'exchange_rate'   => null,
            'guest_token'     => Auth::check() ? null : $guest_token,
            'guest_email'     => $guest_email,
            'refunded'        => 0,
            'refund'          => 0,
            'confirmed'       => 1,
            'items_count'     => count($cart_items) ?? 1,
            'status'          => 'pending',
            'coupon_id'       => $coupon['id'] ?? null,
            'is_subscription' => $subscription ? 1 : 0,
            'products_ids'    => count($cart_items) ? implode(',', array_map(fn($item) => wrap_str($item['item_id']), $cart_items)) : ($subscription ? wrap_str($subscription['id']) : ($prepaid_credit ? wrap_str($prepaid_credit['id']) : null)),
            'order_id'        => null,
            'transaction_id'  => null,
            'payment_url'     => urldecode(Session::pull('short_link')),
            'read_by_admin'   => 0,
            'referrer_id'     => config('referrer_id'),
            'type'            => count($cart_items) ? 'product' : ($subscription ? 'subscription' : 'credits'),
            'sandbox'         => 1,
            'cart_id'         => count($cart_items) ? $request->post('cart_id') : null,
        ];

        if($due_amount > 0)
        {
            $class    = config("payment_gateways.{$payment_gateway['name']}.class") ?? abort(404);
            $class    = "\App\Libraries\\$class";
            $gateway  = new $class();
            $response = $gateway->init_payment($due_amount);

            if($gateway->error_msg ?? null)
            {
                if($return_url)
                {
                    return $gateway::$response === 'json' ? response()->json($gateway->error_msg) : $gateway->error_msg;
                }
                else
                {
                    return $gateway::$response === 'json' ? response()->json($gateway->error_msg) : back()->with($gateway->error_msg);
                }
            }

            $details['currency']      = $gateway->currency_code;
            $details['exchange_rate'] = $gateway->exchange_rate;
            $details['reference_id']  = $gateway->details['reference'];

            $transaction['reference_id']    = $gateway->details['reference'] ?? null;
            $transaction['processor']       = $payment_gateway['name'];
            $transaction['details']         = json_encode($details);
            $transaction['exchange_rate']   = $gateway->exchange_rate;
            $transaction['order_id']        = $gateway->details['order_id'] ?? null;
            $transaction['transaction_id']  = $gateway->details['transaction_id'] ?? null;
            $transaction['payment_url']     = urldecode(Session::pull('short_link'));
                        
            if(!isset($transaction['products_ids']))
            {
                abort(403, __('Product ids col cannot be empty'));
            }

            if($subscription)
            {
                DB::transaction(function() use(&$transaction, $subscription)
                {
                    $transaction = Transaction::create($transaction);

                    User_Subscription::insert([
                        'user_id'         => Auth::id(),
                        'subscription_id' => $subscription['id'],
                        'transaction_id'  => $transaction['id'],
                        'ends_at'         => is_numeric($subscription['days']) && $subscription['days'] > 0
                                             ? date('Y-m-d H:i:s', strtotime(date('Y-m-d H:i:s') . " + {$subscription['days']} days"))
                                             : null,
                        'daily_downloads' => 0,
                        'daily_downloads_date' => $subscription['limit_downloads_per_day'] ? date('Y-m-d') : null
                    ]);
                });
            }
            elseif($prepaid_credit)
            {
                DB::transaction(function() use(&$transaction, $prepaid_credit)
                {
                    $transaction = Transaction::create($transaction);

                    User_Prepaid_Credit::insert([
                        'user_id'            => Auth::id(),
                        'prepaid_credits_id' => $prepaid_credit['id'],
                        'transaction_id'     => $transaction['id'],
                        'credits'            => $prepaid_credit['amount'],
                    ]);
                });
            }
            else
            {
                $transaction = Transaction::create($transaction);
            }


            if($transaction->coupon_id && $transaction->user_id)
            {
                DB::update("UPDATE coupons SET used_by = IF(used_by IS NULL, ?, CONCAT_WS(',', used_by, ?)) WHERE code = ?", 
                ["'{$transaction->user_id}'", "'{$transaction->user_id}'", (string)$coupon['code']]);
            }

            if(count($cart_items))
            {                
                $products_ids = array_column($cart_items, 'item_id');

                $this->update_keys($products_ids, $transaction);
            }

            if($notes = trim($request->input('notes', '')))
            {
                $transaction_note = Transaction_Note::create([
                    'transaction_id' => $transaction->id,
                    'user_id'        => \Auth::id(),
                    'content'        => $notes
                ]);
            }

            if($return_url)
            {
                return $gateway::$response === 'json' ? response()->json($response) : $response;
            }
            else
            {
                if(is_array($response))
                {
                    return $gateway::$response === 'json' ? response()->json($response) : back()->with($response);
                }
                else
                {
                    if($gateway::$response === 'json')
                    {
                        return $request->prepare ? response()->json($response) : redirect()->away($response);
                    }
                    else
                    {
                        return redirect()->away($response);
                    }
                }
            }
        }
        else
        {
            $details['currency']      = config("payments.currency_code");
            $details['exchange_rate'] = 1;
            $details['reference_id']  = generate_transaction_ref();

            $transaction['reference_id']    = $details['reference_id'];
            $transaction['processor']       = 'n-a';
            $transaction['details']         = json_encode($details);
            $transaction['exchange_rate']   = 1;
            $transaction['order_id']        = generate_transaction_ref();
            $transaction['transaction_id']  = generate_transaction_ref();
            $transaction['status']          = 'paid';

            $transaction = Transaction::create($transaction);

            if($transaction->coupon_id && $transaction->user_id)
            {
                DB::update("UPDATE coupons SET used_by = IF(used_by IS NULL, ?, CONCAT_WS(',', used_by, ?)) WHERE code = ?", 
                ["'{$transaction->user_id}'", "'{$transaction->user_id}'", (string)$coupon['code']]);
            }

            if(count($cart_items))
            {
                $products_ids = array_column($cart_items, 'id');

                $this->update_keys($products_ids, $transaction);
            }

            if($transaction->type == 'product')
            {
                $this->order_download_links($transaction, 1);
            }

            return redirect()->route('home.checkout.success')->with([
                "transaction" => $transaction,
                "transaction_status" => "success",
                "transaction_response" => "done"
            ]);
        }
    }



    public function success(Request $request)
    {
        if(session('transaction_status') !== 'success')
        {
            return redirect()->route('home');
        }

        config([
            "meta_data.name"        => config('app.name'),
            "meta_data.title"       => __('Transaction completed'),
            "meta_data.description" => config('app.description'),
            "meta_data.url"         => url()->current(),
            "meta_data.fb_app_id"   => config('app.fb_app_id'),
            "meta_data.image"       => asset('storage/images/'.(config('app.cover') ?? 'cover.jpg')),
        ]);

        $transaction = Session::pull('transaction');

        $transaction->details = is_string($transaction->details) ? json_decode($transaction->details, true) : $transaction->details;

        if($transaction->user_id)
        {
            $transaction->setAttribute('user_email', User::find($transaction->user_id)->email);
        }

        $items = $this->order_download_links($transaction, 0);

        if(isset($transaction->cart_id))
        {
            User_Shopping_Cart_Item::where('uuid', $transaction->cart_id)->delete();
        }

        return view_('checkout_success', ['transaction' => $transaction, 'items' => $items]);
    }



    public function error(Request $request)
    {
        if(session('transaction_status') !== 'error')
        {
            return redirect()->route('home');
        }

        config([
            "meta_data.name"        => config('app.name'),
            "meta_data.title"       => __('Transaction failed'),
            "meta_data.description" => config('app.description'),
            "meta_data.url"         => url()->current(),
            "meta_data.fb_app_id"   => config('app.fb_app_id'),
            "meta_data.image"       => asset('storage/images/'.(config('app.cover') ?? 'cover.jpg')),
        ]);

        $transaction = Transaction::with('user')->where('reference_id', $request->query('ref'))->first();
        $transaction->details = is_string($transaction->details) ? json_decode($transaction->details, true) : $transaction->details;


        if($transaction->user_id)
        {
            $transaction->setAttribute('user_email', User::find($transaction->user_id)->email);
        }

        $items = $this->order_download_links($transaction, 0);

        return view_('checkout_error', ['transaction' => $transaction, 'items' => $items]);
    }



    // WEBHOOK
    public function webhook(Request $request)
    {
      $success     = 0;
      $processor   = mb_strtolower($request->processor);

      $class = config("payment_gateways.{$processor}.class") ?? abort(404);

      $class = "App\Libraries\\$class";
      
      if(!class_exists($class))
      {
        http_response_code($success ? 200 : 400);
        exit;
      }

      $class = new $class;

      if(method_exists($class, "handle_webhook_notif"))
      {
        $response = $class->handle_webhook_notif($request);

        if($response['valid'] ?? null)
        {
          if($response['status'])
          {
            $transaction = $response['transaction'];

            if($transaction->type != 'credits')
            {
              $this->update_affiliate_earnings($transaction);
            }

            if($transaction->status == 'paid')
            {
              $success = 1;

              $transaction->updated_at = now();

              $transaction->save();

              if($transaction->type == 'subscription')
              {
                $this->update_user_subscription_dates($transaction->id);
              }
              elseif($transaction->type == 'credits')
              {
                $this->update_user_prepaid_credits_dates(unwrap_str($transaction->products_ids));
              }

              $this->payment_confirmed_mail_notif($transaction);
            }
          }
        }
      }

      $wh_response = config("payment_gateways.{$processor}.webhook_responses.".($success ? 'success' : 'failed'));

      http_response_code($success ? 200 : 400);

      if(is_array($wh_response))
      {
        return response(json_encode($wh_response), $success ? 200 : 400)->header('Content-Type', 'application/json');
      }
      elseif(is_string($wh_response))
      {
        exit($wh_response);
      }
    }



    // ORDER COMPLETED
    public function order_completed(Request $request)
    {
      if($name = mb_strtolower($request->processor))
      {
          $class       = config("payment_gateways.{$name}.class") ?? abort(404);
          $class       = "App\Libraries\\$class";
          $class       = new $class;

          if(method_exists($class, "complete_payment"))
          {
              $response = $class->complete_payment($request);

              $transaction = $response['transaction'] ?? abort(404);

              if(!is_null($response))
              {
                  if($response['status'])
                  {
                      if(!is_null(config("payment_gateways.{$name}.webhook_responses")) && config('payments.enable_webhooks') === '0')
                      {
                          $transaction->status = 'paid';
                      }

                      if($transaction->type != 'credits')
                      {
                          $this->update_affiliate_earnings($transaction);
                      }

                      if($transaction->status == 'paid')
                      {
                          $transaction->updated_at = now();

                          $transaction->save();

                          if($transaction->type == 'subscription')
                          {
                              $this->update_user_subscription_dates($transaction->id);
                          }
                          elseif($transaction->type == 'credits')
                          {
                              $this->update_user_prepaid_credits_dates(unwrap_str($transaction->products_ids));
                          }

                          $this->payment_confirmed_mail_notif($transaction);
                      }

                      if($transaction->type == 'product')
                      {
                          $this->order_download_links($transaction, 1);
                      }

                      return redirect()->route('home.checkout.success')->with([
                          "transaction" => $transaction,
                          "transaction_status" => "success",
                          "transaction_response" => "done"
                      ]);
                  }
                  else
                  { 
                      return redirect('/')->with(['user_message' => str_ireplace("-", ' ', slug($response['user_message']))]);
                  }
              }
          }
      }

      return redirect('/');
    }




    // NOTIFY BUYER ABOUT THE PAYMENT ONCE IT'S CONFIRMED
    public function payment_confirmed_mail_notif($transaction)
    {
      try
      {
        $buyer_email = $transaction->guest_email ?? User::find($transaction->user_id)->email;

        $transaction_details = json_decode($transaction->details, true);

        $order        = array_merge($transaction->getAttributes(), $transaction_details);
        $products_ids = explode(',', str_replace("'", "", $transaction->products_ids));

        $order_id = $order['order_id'] ?? $order['transaction_id'] ?? $order['reference_id'] ?? null;

        $mail_props = [
          'data'    => $order,
          'action'  => 'send',
          'view'    => 'mail.order',
          'to'      => $buyer_email,
          'subject' => __('Order :number. is completed. Your payment has been confirmed', ['number' => $order_id])
        ];

        sendEmailMessage($mail_props, config('mail.mailers.smtp.use_queue'));

        if(!$transaction->is_subscription)
        {
          Product::whereIn('id', $products_ids)->where('stock', '>', 0)->decrement('stock', 1);
        }

        if(config('app.admin_notifications.sales'))
        {
          $message = [];

          foreach($transaction_details['items'] as $item)
          {
            $message[] = "- {$item['name']}";              
          }

          $message[] = "\n\n<strong>".__('You earned :amount', ['amount' => price($transaction_details['amount'], false)])."</strong>";

          $mail_props = [
              'data'    => [
                  'text' => implode("\n", $message), 
                  'subject' => __('A new sale has been completed by :buyer_email', ['buyer_email' => $buyer_email]),
                  'user_email' => $buyer_email
              ],
              'action'  => 'send',
              'view'    => 'mail.message',
              'to'      => config('app.email'),
              'subject' => __('A new sale has been completed by :buyer_email', ['buyer_email' => $buyer_email])
          ];

          sendEmailMessage($mail_props, config('mail.mailers.smtp.use_queue'));
        }
      }
      catch(\Exception $e){}
    }
    


    public function order_download_links($transaction, $send_email = 1)
    {
        try
        {
            $transaction_details = is_string($transaction->details) ? json_decode($transaction->details, true) : $transaction->details;

            $transaction_details['items'] = array_values($transaction_details['items']);

            if(!isset($transaction_details['items'][0]['item_id']))
            {
                return;
            }

            $products_ids = array_map(fn($item) => $item['item_id'], $transaction_details['items']);

            if(count($products_ids) > 1 && count(array_unique($products_ids, SORT_REGULAR)) === 1)
            {
                $products = new \Illuminate\Database\Eloquent\Collection();

                for($i=0; $i<count($products_ids); $i++)
                {
                    $products->add(Product::find($products_ids[$i]));
                }
            }
            else
            {
                $products = Product::whereIn('id', $products_ids)->get();
            }

            $keys = Key::whereIn('product_id', $products_ids)->where('user_id', $transaction->user_id ?? $transaction->guest_token)->get();

            $download_params = [
                'type'     => null,
                'order_id' => $transaction->id, 
                'user_id'  => $transaction->user_id ?? $transaction->guest_token, 
                'item_id'  => null,
            ];

            $items = [];

            foreach($transaction_details['items'] ?? [] as $k => $item)
            {
                $product = Product::find($item['item_id']);

                if(!$product)
                {
                    continue;
                }

                $download_params['item_id'] = $item['item_id'];

                $items[$k] = (object)[
                    'id'      => $product->id,
                    'name'    => $product->name,
                    'url'     => item_url($product),
                    'license' => null,
                    'file'    => null,
                    'key'     => null,
                    'files'   => 0,
                ];

                $key = $keys->where('product_id', $product->id)->first();

                if(!$key && ($product->file_name || $product->direct_download_link || config('app.generate_download_links_for_missing_files')))
                {
                    $items[$k]->file = route('home.download', array_merge($download_params, ['type' => 'file']));
                    $items[$k]->files += 1;
                }

                if($product->enable_license && isset($item['license']))
                {
                    $enc_license = encrypt(json_encode([
                        'name'    => $item['extended_license'] ? __('Extended license') : __('Regular license'), 
                        'license' => $item['license']
                    ]), false);

                    $items[$k]->license = route('home.download', array_merge($download_params, ['type' => 'license', 'content' => $enc_license]));
                    $items[$k]->files += 1;
                }

                if($key)
                {
                    $items[$k]->key = route('home.download', array_merge($download_params, ['type' => 'key', 'content' => encrypt($key->code, false)]));
                    $items[$k]->files += 1;
                }
            }

            foreach($items as $k => $item)
            {
                if($item->files === 0)
                {
                    unset($items[$k]);
                }
            }

            if($send_email)
            {
                if(!count($items))
                {
                    return [];
                }

                $order_id     = $transaction->reference_id;
                $buyer_email  = $transaction->guest_email ?? User::find($transaction->user_id)->email;

                $mail_props = [
                    'data'    => ['items' => $items, 'order_id' => $order_id],
                    'action'  => 'send',
                    'view'    => 'mail.download_links',
                    'to'      => $buyer_email,
                    'subject' => __(':app_name - Download links for order number :order_id', ['app_name' => config('app.name'), 'order_id' => $order_id])
                ];

                sendEmailMessage($mail_props, config('mail.mailers.smtp.use_queue'));
            }

            return $items;
        }
        catch(\Exception $e)
        {

        }
    }


    
    public function update_keys($products_ids, $transaction)
    {
        $products_ids = array_filter($products_ids);
        
        foreach($products_ids as $product_id)
        {
            if($key = Key::useIndex('product_id')->where('product_id', $product_id)->where('user_id', null)->first())
            {
                DB::update("UPDATE key_s SET user_id = ?, purchased_at = ? WHERE id = ?", [$transaction->user_id ?? $transaction->guest_token, now()->format('Y-m-d H:i:s'), $key->id]);
            }
        }
    }



    public function update_affiliate_earnings($transaction)
    { 
        if($transaction->referrer_id && $transaction->type != "credits")
        {
          Affiliate_Earning::insert([
              'referrer_id'         => $transaction->referrer_id,
              'referee_id'          => $transaction->user_id ?? $transaction->guest_token,
              'transaction_id'      => $transaction->id,
              'commission_value'    => format_amount($transaction->amount * config('affiliate.commission', 0) / 100),
              'commission_percent'  => config('affiliate.commission', 0),
              'amount'              => $transaction->amount,
              'paid'                => 0,
          ]);
        }
    }



    public function update_user_subscription_dates(int $transaction_id) 
    {
      DB::update("UPDATE user_subscription 
                    JOIN transactions ON user_subscription.transaction_id = transactions.id
                    SET ends_at = DATE_ADD(ends_at, INTERVAL TIMESTAMPDIFF(SECOND, transactions.created_at, NOW()) SECOND)
                    WHERE user_subscription.transaction_id = ? AND user_subscription.ends_at IS NOT NULL", [$transaction_id]);

      DB::update("UPDATE user_subscription 
                    JOIN transactions ON user_subscription.transaction_id = transactions.id
                    SET daily_downloads_date = DATE_FORMAT(NOW(), '%Y-%m-%d')
                    WHERE user_subscription.transaction_id = ? AND user_subscription.daily_downloads_date IS NOT NULL", [$transaction_id]);
    }


    public function update_user_prepaid_credits_dates(int $prepaid_credits_id)
    {
      DB::update("UPDATE user_prepaid_credits SET updated_at = DATE_ADD(updated_at, INTERVAL TIMESTAMPDIFF(SECOND, updated_at, NOW()) SECOND) WHERE id = ?", 
        [$prepaid_credits_id]);
    }


    public function update_pending_transactions()
    {
      if(config('payments.update_pending_transactions') === 0)
      {
        abort(404);
      }

      $transactions = Transaction::where(['status' => 'pending', 'refunded' => 0, 'confirmed' => 1])->get();
      
      foreach($transactions as $transaction)
      {
        try
        {
          if(config("payments_gateways.{$transaction->processor}"))
          {
            $payment_class = config("payment_gateways.{$transaction->processor}.class");
            $payment_class = "\App\Libraries\\$payment_class";
            $payment_class = new $payment_class;

            if(!method_exists($payment_class, "handle_webhook_notif") && method_exists($payment_class, 'verify_payment'))
            {
              $response = $payment_class->verify_payment($transaction);

              if($response['status'] ?? null === true)
              {
                $transaction->status = 'paid';

                $transaction->save();
              }
            }
          }
        }
        catch(\Throwable $t)
        {

        }
      } 
    }
}
