<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\{ Product, Key, User_Subscription, Pricing_Table, Page, Post, Setting, Newsletter_Subscriber, Reaction, Search, Category_Product,
  Subscription_Same_Item_Downloads, Prepaid_Credit, Category, Support_Email, Support, Review, Comment, Faq, Notification, Transaction, User, User_Prepaid_Credit, Custom_Route, Affiliate_Earning, Cashout, Coupon, Temp_Direct_Url, User_Shopping_Cart_Item };
use Illuminate\Support\Facades\{ DB, File, Hash, Validator, Config, Auth, Mail, Cache, Session, View };
use App\Libraries\{ DropBox, GoogleDrive, IyzicoLib, YandexDisk, OneDrive, AmazonS3, Wasabi, Paypal };
use ZipArchive;
use GeoIp2\Database\Reader;
use Intervention\Image\Facades\Image;
use BrowserDetect;
use Illuminate\Support\{ Carbon, Str };
use Illuminate\Pagination\{ Paginator, LengthAwarePaginator };
use Intervention\Image\ImageManager;

class HomeController extends Controller
{
    public function __construct()
    {
        if(config('app.installed') === true)
        {
            config([
                "meta_data.name" => config('app.name'),
                "meta_data.title" => config('app.title'),
                "meta_data.description" => config('app.description'),
                "meta_data.url" => url()->current(),
                "meta_data.fb_app_id" => config('app.fb_app_id'),
                "meta_data.image" => asset('storage/images/'.(config('app.cover') ?? 'cover.jpg'))
            ]);

            $this->middleware('maintenance_mode');

            View::share(['payment_gateways' => array_values(config('payments_gateways', []))]);

            if(!cache('counters'))
            {
                $fake_counters = [
                    "orders" => config('app.fake_counters') ? rand(950, 1000) : Transaction::count(),
                    "products" => config('app.fake_counters') ? rand(1000, 2000) : Transaction::count(),
                    "categories" => config('app.fake_counters') ? rand(100, 200) : Category::count(),
                    "affiliate_earnings" => price(config('app.fake_counters') ? rand(5000, 10000) : Affiliate_Earning::sum('amount'), 0, 0),
                ];

                Cache::put("counters", $fake_counters, now()->addDays(1));
            }

            $template = config('app.template');

            if($template === 'tendra')
            {
                if(Cache::has('mega_menu') && count(Cache::get('mega_menu', [])))
                {
                    $mega_menu = cache('mega_menu', []);
                }
                else
                {
                    $mega_menu = [];

                    $products = DB::select(<<<QRY
                        WITH RankedProducts AS (
                          SELECT id, category as category_id, cover, rating, fake_rating, sales, slug, short_description, `name`, price, ROW_NUMBER() OVER (PARTITION BY category ORDER BY id) AS row_num
                          FROM products WHERE products.active = 1
                        )
                        SELECT id, category_id, `name`, cover, slug, short_description, price FROM RankedProducts WHERE row_num <= 7;
                    QRY);

                    $grouped_products = collect($products)->groupBy('category_id');

                    foreach($grouped_products as $category_id => $products)
                    {
                        $mega_menu[$category_id] = [
                          'category_slug' => config("categories.category_parents.{$category_id}.slug"),
                          'category_name' => config("categories.category_parents.{$category_id}.name"),
                          'category_id'   => config("categories.category_parents.{$category_id}.id"),
                          'items'         => $products->map(fn($product) => (array)$product)->toArray(),
                        ];
                    }

                    Cache::put('mega_menu', $mega_menu, now()->addMinutes(5));
                }

                View::share(['mega_menu' => $mega_menu]);
            }
        }
    }


    public $product_columns = [
        'products.id',
        'products.name',
        'products.slug',
        'products.cover',
        'products.price',
        'products.promo_price',
        'products.promo_time',
        'products.short_description',
        'products.sales',
        'products.rating',
        'products.category as category_id',
        'products.release_date',
        'products.for_subscriptions',
        'products.extended_price',
        'products.created_at',
        'products.minimum_price',
        'products.enable_license',
    ];


    public function index()
    {
        $template = config('app.template', 'axies');

        $this->product_columns = array_merge($this->product_columns, [
            DB::raw('(products.promo_time IS NOT NULL and products.promo_time > UNIX_TIMESTAMP()) as has_promo'),
            DB::raw('products.price = 0 as is_free'),
        ]);

        config([
          'json_ld' => [
            '@context' => 'http://schema.org',
            '@type' => 'WebSite',
            'name' => config('app.name'),
            'url' => config('app.url'),
            'potentialAction' => [
              '@type' => 'SearchAction',
              'target' => route("home.products.q").'?&q={query}',
              'query' => 'required'
            ]
          ]
        ]);
        
        return call_user_func([$this, "{$template}_home"]);
    }


    private function axies_home()
    {
        $request_id = "axies_home";
        $data       = null;

        if(config('app.enable_data_cache'))
        {
            $data = Cache::get($request_id);
        }
        else
        {
            Cache::forget($request_id);
        }

        if(!$data || !config('app.enable_data_cache'))
        {
            $posts = Post::useIndex('primary')->where('active', 1)->orderBy('id', 'DESC')->limit(4)->get();
            
            foreach($posts as &$post)
            {
                $post->setAttribute('category', config("categories.posts.{$post->category}"));
            }

            $featured_products  = $this->trending_products(6);
            $trending_products  = $this->trending_products(7);
            $newest_products    = $this->newest_products(14);
            $free_products      = $this->free_products(4);
            $random_products    = $this->products_by_status(status:'random', limit:4, category_id:null, randomize:true);

            $data = compact('posts', 'newest_products', 'featured_products', 'trending_products', 'free_products', 'random_products');
            
            Cache::put($request_id, $data, now()->addSeconds(3600));
        }

        return view_('home', $data);
    }


    private function tendra_home()
    {
        $request_id = "tendra_home";
        $data       = null;

        if(config('app.enable_data_cache'))
        {
            $data = Cache::get($request_id);
        }
        else
        {
            Cache::forget($request_id);
        }

        if(!$data || !config('app.enable_data_cache'))
        {
            $posts = Post::useIndex('primary')->where('active', 1)->orderBy('id', 'DESC')->limit(4)->get();

            foreach($posts as &$post)
            {
                $post->setAttribute('category', config("categories.posts.{$post->category}"));
            }


            $featured_products = [];

            foreach(config('categories.category_parents') as $parent_category)
            { 
                $featured_products[$parent_category['slug']] = $this->featured_products(8, $parent_category['id']);
            }

            $featured_products =  array_filter($featured_products, function($items, $k)
                                  {
                                      return $items->count();
                                  }, ARRAY_FILTER_USE_BOTH);

            $newest_products = $this->newest_products(10);

            $subscriptions   = Pricing_Table::useIndex('position')->orderBy('position', 'asc')->get();

            foreach($subscriptions as &$subscription)
            {
              $subscription->specifications = json_decode($subscription->specifications, false, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES) ?? (object)[];
            }

            $data = compact('posts', 'newest_products', 'featured_products', 'subscriptions');
            
            Cache::put($request_id, $data, now()->addSeconds(3600));
        }

        return view_('home', $data);
    }



    public function featured_products($limit = 15, $category_id = null, $randomize = false)
    {
        return $this->products_by_status('featured', $limit, $category_id, $randomize);
    }


    public function trending_products($limit = 15, $category_id = null, $randomize = false)
    {
        return $this->products_by_status('treding', $limit, $category_id, $randomize);
    }


    public function newest_products($limit = 15)
    {
        $products = Product::select($this->product_columns)->forceIndex('active')->where('active', 1)->orderBy('id', 'desc')->limit($limit)->get();

        foreach($products as &$product)
        {
          $product->setAttribute('category', config("categories.category_parents.{$product->category_id}"));
        }

        return $products;
    }


    public function free_products($limit, $randomize = false)
    {
        $products = Product::select($this->product_columns)->forceIndex('active')
        ->whereRaw('promo_price = 0 AND (promo_time IS NULL OR promo_time > UNIX_TIMESTAMP(NOW()))')
        ->orderBy('id', 'desc')->limit($limit);

        if($randomize)
        {
          $products = $products->orderByRaw('RAND()');
        }

        $products = $products->get();

        foreach($products as &$product)
        {
          $product->setAttribute('category', config("categories.category_parents.{$product->category_id}"));
        }

        return $products;
    }


    public function products_by_status($status, $limit = 15, $category_id = null, $randomize = false)
    {
        $status = mb_strtolower($status);

        $status = preg_match('/^trending|featured$/i', $status) ? $status : 'featured';

        $product_ids = Category_Product::select('product_id');

        if($status !== 'random')
        {
            $product_ids = $product_ids->where($status, 1);
        }

        if($category_id)
        {
            $product_ids = $product_ids->where('category_id', $category_id);
        }

        if($randomize)
        {
            $product_ids = $product_ids->orderByRaw('RAND()');
        }

        $product_ids = $product_ids->limit($limit)->get()->pluck('product_id')->toArray();

        if(!count($product_ids))
        {
            return collect([]);  
        }

        $products = Product::forceIndex('primary')
                    ->select($this->product_columns)
                    ->where(['products.active' => 1])
                    ->whereIn('products.id', $product_ids)->get();

        foreach($products as &$product)
        {
            $product->setAttribute('category', config("categories.category_parents.{$category_id}"));
        }

        return $products;
    }


    public function affiliate()
    {
        return view('front.affiliate');
    }


    // Single page
    public function page($slug)
    {        
        if(!$page = Page::useIndex('slug', 'active')->where(['slug' => $slug, 'active' => 1])->first())
          abort(404);

        $page->increment('views', 1);

        config([
          "meta_data.name" => config('app.name'),
          "meta_data.title" => $page->name,
          "meta_data.description" => $page->short_description,
          "json_ld" => [
            '@context' => 'http://schema.org',
            '@type' => 'WebPage',
            'name' => $page->name,
            'description' => $page->short_description,
          ]
        ]);

        return view_('page',compact('page'));
    }



    // Products per category
    public function products(Request $request)
    { 
        if($sort = strtolower($request->query('sort')))
        {
            preg_match('/^(?P<sort>relevance|price|rating|featured|trending|newest)_(?P<order>asc|desc)$/i', $sort, $matches) || abort(404);

            extract($matches);
        }
        else
        {
            list($sort, $order) = ['id', 'desc'];
        }

        $sort = $sort === 'newest' ? 'updated_at' : $sort;
        $sort = $sort === 'relevance' ? 'id' : $sort;

        $categories    = config('categories.category_parents', []);
        $subcategories = config('categories.category_children', []);

        $category      = null;
        $subcategory   = null;
        
        $products = Product::select('id')->where('products.active', 1);

        if($request->category_slug)
        {
            $category = array_filter(config('categories.category_parents'), function($category) use ($request)
            {
                return strtolower($category['slug']) === strtolower($request->category_slug);
            }, 0) ?? abort(404);

            $category = array_shift($category);

            $products = $products->useIndex('category')->where('category', $category['id']);
        }

        if($request->subcategory_slug && $category)
        {
            $subcategory = array_filter(config("categories.category_children.{$category['id']}"), function($subcategory) use ($request)
            {
                return strtolower($subcategory['slug']) === strtolower($request->subcategory_slug);
            }, 0) ?? abort(404);

            $subcategory = array_shift($subcategory);

            $products = $products->useIndex('category')->where('subcategories', 'like', "'%{$subcategory['id']}%'");
        }


        if($filter = strtolower($request->filter))
        {
            if($filter === 'free')
            {
                $products = $products->where(function ($query)
                            {
                              $query->useIndex('price')->where('price', 0)
                                    ->orWhereRaw("CURRENT_DATE between substr(free, 10, 10) and substr(free, 28, 10)");
                            });
            }
            elseif($filter === 'trending')
            {
                $products = $products->useIndex('trending')->where("products.trending", 1);

                $sort = 'trending';
                $order = 'desc';
            }
            elseif($filter === 'featured')
            {
                $products = $products->useIndex('featured')->where("products.featured", 1);

                $sort  = 'featured';
                $order = 'desc';
            }
            elseif($filter === 'flash')
            {
                $products = Product::useIndex('promo_price')->where("products.promo_price IS NOT NULL");

                $sort  = 'price';
                $order = 'asc';
            }
            elseif($filter === 'newest')
            {

            }
        }


        if($q = trim($request->query('q')))
        {
            $search = new Search;
            
            $search->keywords = $q;
            $search->user_id  = Auth::id();

            $search->save();

            $products = $products->whereRaw('(products.name LIKE ? OR products.slug LIKE ? OR products.tags LIKE ?)', ["%{$q}%", "%{$q}%", "%{$q}%"]);

            config([
                "meta_data.title"       => config('app.name').' - '.__('Searching for').' '.ucfirst($request->q),
                "meta_data.description" => $category->description ?? config('meta_data.description')
            ]);
        }

        if($rating = trim($request->query('rating')))
        {
            if(is_numeric($rating))
            {
                $products = $products->where('rating', ">=", $rating); 
            }
        }

        if($tags = $request->query('tags'))
        {
            $tags = implode('|', array_filter(explode(',', $tags)));

            $products = $products->where(['products.tags', 'REGEXP', $tags]);
        }


        $cities = $country = null;

        if(config('app.products_by_country_city'))
        {
            if($country = $request->query('country'))
            {
                if($cities = urldecode($request->query('cities')))
                {
                    if($cities = array_filter(explode(',', $cities)))
                    {
                        $cities   = implode('|', $cities);
                        
                        $products = $products->whereRaw('country_city REGEXP ?', ['^\{"country":"'. $country .'","city":"'. $cities .'"\}$']);

                        $cities   = str_ireplace('|', ',', $cities);
                    }
                    else
                    {
                        $cities = null;
                    }
                }
                else
                {
                    $products = $products->whereRaw('country_city REGEXP ?', ['^\{"country":"'. $country .'","city":.*\}$']);
                }
            }
        }

        if($price_range = $request->query('price'))
        {
            $price_range = json_decode($price_range, true) ?? abort(404);

            if($price_range[0] > $price_range[1] || count(array_filter($price_range, fn($price) => is_numeric($price))) !== 2)
            {
                return back();
            }

            $products = $products->whereBetween('products.price', [$price_range[0], $price_range[1]]);

            $price_range = array_combine(['min', 'max'], $price_range);
        } 
        

        $selected_included_files = [];
        
        if(intval(config('app.filter_by_included_files')))
        {
            if($selected_included_files = array_filter(explode(',', $request->query('included-files', ''))))
            {
                $products = $products->where(function($query) use($selected_included_files)
                {
                    if(count($selected_included_files) === 1)
                    {
                        $query->where('products.included_files', 'LIKE', "%{$selected_included_files[0]}%");  
                    }
                    else
                    {
                        $query->where('products.included_files', 'LIKE', "%{$selected_included_files[0]}%");

                        foreach(array_slice($selected_included_files, 1) as $included_file)
                        {
                            $query->orWhere('products.included_files', 'LIKE', "%{$included_file}%");
                        }
                    }
                });
            }
        }

        $products_base = $products->orderBy($sort, $order)->paginate(config('app.items_per_page', 20));

        $products_collection = $products_base->getCollection();

        $product_ids = $products_collection->pluck('id')->toArray();

        $products = Product::useIndex('primary')->select($this->product_columns)->whereIn('id', $product_ids)->orderBy($sort, $order)->get();

        $total = $products_base->total();

        $per_page = config('app.items_per_page', 20);

        $current_page = $request->query("page") ?? 1;

        $starting_point = ($current_page * $per_page) - $per_page;

        $products = new LengthAwarePaginator($products, $total, $per_page, $current_page, [
            'path'  => $request->url(),
            'query' => $request->query(),
        ]);

        $tags = [];

        foreach($products->items() as &$item)
        {
            $tags = array_merge($tags, array_filter(array_map('trim', explode(',', $item->tags))));
        }

        $tags = array_unique($tags);

        config([
          'json_ld' => [
            '@context' => 'https://schema.org',
            '@type' => 'ItemList',
            'url' => url()->full(),
            'numberOfItems' => $products->total(),
            'itemListElement' => array_reduce($products->items(), function($carry, $item)
            {
              $carry[] = [
                '@type' => 'Product',
                'image' => asset("storage/covers/{$item->cover}"),
                'url'   => item_url($item),
                'name'  => $item->name,
                'offers' => [
                  '@type' => 'Offer',
                  'price' => $item->price,
                ],
              ];

              return $carry;
            }, [])
          ]
        ]);

        $ratings = [];

        if(config('app.template') === "tendra")
        {
            $ratings[5] = DB::select("SELECT COUNT(id) AS ratings FROM products USE INDEX(fake_rating) WHERE fake_rating >= 5")[0]->ratings ?? 0;
            $ratings[4] = DB::select("SELECT COUNT(id) AS ratings FROM products USE INDEX(fake_rating) WHERE fake_rating BETWEEN 4 AND 4.99")[0]->ratings ?? 0;
            $ratings[3] = DB::select("SELECT COUNT(id) AS ratings FROM products USE INDEX(fake_rating) WHERE fake_rating BETWEEN 3 AND 3.99")[0]->ratings ?? 0;
            $ratings[2] = DB::select("SELECT COUNT(id) AS ratings FROM products USE INDEX(fake_rating) WHERE fake_rating BETWEEN 2 AND 2.99")[0]->ratings ?? 0;
            $ratings[1] = DB::select("SELECT COUNT(id) AS ratings FROM products USE INDEX(fake_rating) WHERE fake_rating BETWEEN 1 AND 1.99")[0]->ratings ?? 0;
        }

        $sort = $sort === 'id' ? __('Best match') : $sort;
        $sort = mb_ucfirst(str_ireplace('_', ' ', $sort));

        $included_files = [];

        if(intval(config('app.filter_by_included_files')))
        {
            $included_files = DB::select("SELECT GROUP_CONCAT(included_files SEPARATOR ',') as types FROM products WHERE FIND_IN_SET(id, ?)", [implode(',', $product_ids)])[0];

            $included_files = $included_files->types ?? '';
            $included_files = explode(',', $included_files);
            $included_files = array_unique($included_files, SORT_NATURAL);

            sort($included_files);
        }

        if(count($selected_included_files))
        {
            $selected_included_files = array_unique($selected_included_files, SORT_NATURAL);

            $selected_included_files = array_combine(array_values($selected_included_files), $selected_included_files);
        }

        return view_('products', compact('products', 'tags', 'country', 'cities', 'ratings', 'category', 'subcategory', 'price_range', 'rating', 'sort', 'included_files', 'selected_included_files'));
    }


    public function live_search(Request $request)
    {
        $products = [];

        if($q = $request->post('q'))
        {
          $products = DB::select("SELECT id, `name`, slug, cover FROM products WHERE active = 1 AND (`name` LIKE ? OR slug LIKE ?) LIMIT 5", ["%{$q}%", "%{$q}%"]);

          foreach($products as &$product)
          {
              $product->cover = src("storage/covers/{$product->cover}");
          }
        }

        return response()->json(compact('products'));
    }



    // Single product
    public function product(Request $request)
    {
      $user_id = Auth::id();

      $request_id = strtok($request->server('REQUEST_URI'), '?');
      $data       = null;
      
      if(intval(config('app.enable_data_cache')))
      {
        $data = Cache::get($request_id);
      }

      if(!$data || !intval(config('app.enable_data_cache')))
      {
        $product =  $request->via_permalink 
                    ? (Product::where('permalink', $request->slug)->first() ?? abort(404))
                    : (Product::withCount('comments', 'reviews')->find($request->id) ?? abort(404));

        if(!$request->via_permalink && urldecode($request->slug) !== mb_strtolower(urldecode($product->slug)))
        {
            return redirect(item_url($product));
        }
      
        $category           = Category::find($product->category) ?? abort(404);
        $has_keys           = Key::where('product_id', $product->id)->exists() ? 1 : 0;
        $remaining_keys     = Key::where(['product_id' => $product->id, 'user_id' => null])->count();

        $product->table_of_contents = json_decode($product->table_of_contents ?? '', true) ?? [];

        $product->fill([
            "remaining_keys"               => $remaining_keys,
            "has_keys"                     => $has_keys,
            "is_free"                      => $product->price <= 0, 
            "category"                     => $category,
            "sales"                        => $product->sales,
            "reviewed"                     => false,
            "purchased"                    => false,
            "comments_count"               => $product->comments_count,
            "reviews_count"                => $product->reviews_count,
            "remaining_downloads"          => null,
            "valid_subscription"           => 0,
            "limit_downloads_reached"      => 0,
            "daily_download_limit_reached" => 0,
            "stock"                        => $has_keys ? $remaining_keys : $product->stock,
        ]);

        $reviews =  Review::useIndex('product_id', 'approved')
                    ->selectRaw("reviews.*, users.name, SUBSTR(users.email, 1, LOCATE('@', users.email)-1) as alias_name, CONCAT(users.firstname, ' ', users.lastname) AS fullname, IFNULL(users.avatar, 'default.webp') AS avatar")
                    ->leftJoin('users', 'users.id', '=', 'reviews.user_id')
                    ->where(['reviews.product_id' => $product->id, 'reviews.approved' => 1])
                    ->orderBy('created_at', 'DESC')->get();

        if($fake_reviews = json_decode($product->fake_reviews))
        {
            $last_id = max($reviews->pluck('id')->toArray() ?: [0]);

            foreach($fake_reviews as &$fake_review)
            {
                $last_id++;

                $fake_review->id         = $last_id;
                $fake_review->name       = $fake_review->username;
                $fake_review->avatar     = "default.webp";
                $fake_review->created_at = format_date($fake_review->created_at, 'Y-m-d H:i:s');
                $fake_review->updated_at = $fake_review->created_at;
                $fake_review->parent     = null;
                $fake_review->content    = $fake_review->review;
                $fake_review->product_id = $product->id;
                $fake_review->user_id    = md5($fake_review->name);
                $fake_review->approved   = 1;
                $fake_review->rating     = $fake_review->rating;
                $fake_review->is_admin   = 0;

                unset($fake_comment->username, $fake_review->review);

                $fake_review = new Review((array)$fake_review);
            }

            $reviews = $reviews->merge(collect($fake_reviews))->sortByDesc('created_at');
            $ratings = $reviews->pluck('rating')->toArray();

            $product->rating = $ratings ? round(array_sum($ratings) / count($ratings)) : 0;
        }

        $comments = Comment::useIndex('product_id', 'approved')
                    ->selectRaw("comments.*, users.name, SUBSTR(users.email, 1, LOCATE('@', users.email)-1) as alias_name, CONCAT(users.firstname, ' ', users.lastname) AS fullname, IFNULL(users.avatar, 'default.webp') AS avatar, IF(users.role = 'admin', 1, 0) as is_admin, 
                      IF((SELECT COUNT(transactions.id) FROM transactions WHERE transactions.user_id = comments.user_id AND transactions.status = 'paid' AND transactions.refunded = 0 AND transactions.confirmed = 1 AND transactions.products_ids REGEXP CONCAT('\'', comments.product_id, '\'')) > 0, 1, 0) as item_purchased")
                    ->leftJoin('users', 'users.id', '=', 'comments.user_id')
                    ->where(['comments.product_id' => $product->id, 'comments.approved' => 1])
                    ->orderBy('id', 'ASC')->get();

        if($fake_comments = json_decode($product->fake_comments))
        {
          $product->comments_count = $comments->count() + count($fake_comments);

          $last_id = max($comments->pluck('id')->toArray() ?: [0]);

          foreach($fake_comments as &$fake_comment)
          {
            $last_id++;

            $fake_comment->id         = $last_id;
            $fake_comment->name       = $fake_comment->username;
            $fake_comment->avatar     = "default.webp";
            $fake_comment->created_at = format_date($fake_comment->created_at, 'Y-m-d H:i:s');
            $fake_comment->updated_at = $fake_comment->created_at;
            $fake_comment->parent     = null;
            $fake_comment->body       = $fake_comment->comment;
            $fake_comment->product_id = $product->id;
            $fake_comment->user_id    = md5($fake_comment->name);
            $fake_comment->approved   = 1;
            $fake_comment->item_purchase = null;
            $fake_comment->is_admin   = 0;

            unset($fake_comment->username, $fake_comment->comment);

            $fake_comment = new Comment((array)$fake_comment);
          }

          $comments = $comments->merge(collect($fake_comments))->sortByDesc('created_at');

          $product->comments_count = $comments->count();
        }

        $similar_products = Product::useIndex('primary', 'category', 'active')
                            ->select($this->product_columns)
                            ->where(['products.category' => $product->category['id'], 
                                     'products.active' => 1,
                                     'products.for_subscriptions' => 0])
                            ->where('products.id', '!=', $product->id)
                            ->orderByRaw('rand()')
                            ->limit(4)->get();

        foreach($similar_products as $similar_product)
        {
            $similar_product->setAttribute('category', config("categories.category_parents.{$similar_product['category_id']}"));
        }

        if($parents = $comments->where('parent', null)->sortByDesc('created_at')) // parents comments only
        {
          $children = $comments->where('parent', '!=', null); // children comments only

          // Append children comments to their parents
          $parents->map(function($item, $key) use ($children, $request, $product)
          {
            $request->merge(['item_type' => 'comment', 'item_id' => $item->id, 'product_id' => $product->id]);

            $item->reactions = $this->get_reactions($request); 

            $item->children = $children->where('parent', $item->id)->sortBy('created_at');

            foreach($item->children as $children)
            {
              $request->merge(['item_type' => 'comment', 'item_id' => $children->id, 'product_id' => $product->id]);

              $children->reactions = $this->get_reactions($request); 
            }
          });
        }

        if($product->country_city)
        {
          $country_city = json_decode($product->country_city);

          $product->country = $country_city->country ?? null;
          $product->city = $country_city->city ?? null;
        }

        if($product->screenshots)
        {
          $product->screenshots = array_reduce(explode(',', $product->screenshots), function($ac, $img)
                                  {
                                    $ac[] = src("storage/screenshots/{$img}");
                                    return $ac;
                                  }, []);
        }

        $product->tags = array_filter(explode(',', $product->tags));

        $product->additional_fields = json_decode($product->additional_fields);

        $product->faq = json_decode($product->faq, true) ?? [];

        if(count(array_column($product->faq, 'Q')))
        {
          $faqs = [];

          foreach($product->faq as $faq)
          {
            $faqs[] = [
              'question' => $faq['Q'] ?? '',
              'answer' => $faq['A'] ?? ''
            ];
          }

          $product->faq = $faqs;
        }

        $product->faq = arr2obj($product->faq);

        $subscriptions =  Pricing_Table::whereRaw("CASE 
                            WHEN pricing_table.products IS NOT NULL 
                              THEN FIND_IN_SET(?, pricing_table.products)
                            ELSE 1=1
                          END", [$product->id])->get();

        $product->cover = $product->cover ?? "default.webp";

        $json_ld = [
          "@context" => "https://schema.org/",
          "@type" => "Product",
          "name" => $product->name,
          "image" => route('resize_image', ['name' => pathinfo($product->cover, PATHINFO_FILENAME), 'size' => 256, "ext" => pathinfo($product->cover, PATHINFO_EXTENSION)]),
          "description" => $product->short_description,
          "aggregateRating" => [
            "@type" => "AggregateRating",
            "ratingValue" => $product->fake_rating ?? $product->rating ?? 0,
            "reviewCount" => $product->reviews_count ?? 0,
          ],
          "offers" => [
            "@type" => "Offer",
            "availability" => "https://schema.org/".(out_of_stock($product) ? 'OutOfStock' : 'InStock'),
            "price" => price($product->price),
            "priceCurrency" => config('payments.currency_code'),
          ],
        ];

        if(strtolower(pathinfo($product->cover, PATHINFO_EXTENSION)) === "svg")
        {
          unset($json_ld["image"]); 
        }

        if($product->for_subscriptions)
        {
          unset($json_ld["offers"]);
        }

        $data = [
          'subscriptions' => $subscriptions,
          'title'     => mb_ucfirst($product->name),
          'product'   => $product,
          'reviews'   => $reviews,
          'comments'  => $parents, // Parents comments with their children
          'similar_products' => $similar_products,
          'json_ld' => $json_ld
        ];

        Cache::put($request_id, $data, now()->addSeconds(3600));
      }

      extract($data);

      if(Auth::check())
      {
        $subscriptions = User_Subscription::where('user_id', Auth::check())->get();

        $subscription = User_Subscription::useIndex('user_id', 'subscription_id')
                        ->selectRaw("
                          pricing_table.limit_downloads_same_item, user_subscription.transaction_id,
                          (user_subscription.ends_at IS NOT NULL AND user_subscription.ends_at < CURRENT_TIMESTAMP) as time_expired,
                          (pricing_table.limit_downloads > 0 AND user_subscription.downloads >= pricing_table.limit_downloads) as limit_downloads_reached,
                          (pricing_table.limit_downloads_per_day > 0 AND user_subscription.daily_downloads >= pricing_table.limit_downloads_per_day AND user_subscription.daily_downloads_date = CURDATE()) as daily_download_limit_reached,
                          IF(pricing_table.limit_downloads_same_item > 0, pricing_table.limit_downloads_same_item - IFNULL(subscription_same_item_downloads.downloads, 0), null) as remaining_downloads,
                          (pricing_table.limit_downloads_same_item > 0 AND subscription_same_item_downloads.downloads >= pricing_table.limit_downloads_same_item) as same_items_downloads_reached")
                          ->join('pricing_table', 'user_subscription.subscription_id', '=', 'pricing_table.id')
                          ->join('products', 'products.id', '=', DB::raw($product->id))
                          ->leftJoin(DB::raw('subscription_same_item_downloads USE INDEX(product_id, subscription_id)'), function($join) use($product)
                          {
                            $join->on('subscription_same_item_downloads.subscription_id', '=', 'user_subscription.id')
                                 ->where('subscription_same_item_downloads.product_id', $product->id);
                          })
                          ->join(DB::raw('transactions USE INDEX(primary)'), 'user_subscription.transaction_id', '=', 'transactions.id')
                          ->where(function($query)
                          {
                              $query->where('transactions.refunded', 0)
                                    ->orWhere('transactions.refunded', null);
                          })
                          ->where(['transactions.status' => 'paid', 'user_subscription.user_id' => Auth::id()])
                          ->whereRaw('CASE 
                                        WHEN pricing_table.products IS NOT NULL
                                          THEN FIND_IN_SET(?, pricing_table.products)
                                        ELSE 
                                          1=1
                                      END
                                        ', [$product->id])
                          ->having('time_expired', 0)
                          ->first();

        if($subscription)
        {
            $product->transaction_id               = $subscription->transaction_id;
            $product->valid_subscription           = 1;
            $product->remaining_downloads          = $subscription->remaining_downloads;
            $product->limit_downloads_reached      = $subscription->limit_downloads_reached;
            $product->daily_download_limit_reached = $subscription->daily_download_limit_reached;

            if($subscription->limit_downloads_reached || $subscription->daily_download_limit_reached)
            {
              $product->valid_subscription = 0;
            }
        }
      }
      
      $product->reviewed  = Auth::check() ? (Review::where(['product_id' => $product->id, 'user_id' => Auth::id()])->exists() ? 1 : 0) : 0;

      $puchase_conds = ['confirmed' => 1, 'status' => 'paid', 'refunded' => 0];

      if(!config('app.allow_download_in_test_mode'))
      {
        $puchase_conds['sandbox'] = 0;
      }

      $product->fill(['purchased' => 0, 'order_id' => null]);

      if(Auth::check())
      {
        $order = (Transaction::where((array_merge($puchase_conds, ['user_id' => Auth::id()])))->where('products_ids', 'LIKE', "%'{$product->id}'%")->first());
        
        $product->purchased = $order ? 1 : 0;
        $product->order_id = $order->id ?? null;
      }
      
      if(!$product->purchased && !$product->valid_subscription)
      {
        if($guest_token = $request->query('guest_token'))
        {
          $product->purchased = Transaction::useIndex('guest_token')
                                ->where(['guest_token' => $guest_token, 'status' => 'paid', 'refunded' => 0, 'confirmed' => 1])
                                ->where('transactions.products_ids', 'LIKE', "'%{$product->id}%'")
                                ->exists();
        }
      }

      if($request->query('delete_review_id') && $request->query('type') === 'reviews')
      {
        Review::where(['product_id' => $product->id, 'user_id' => Auth::id(), 'id' => $request->delete_review_id])->delete();

        return redirect(url()->current().'?tab=reviews');
      }
      elseif($request->query('delete_comment_id') && $request->query('type') === 'comments')
      {
        if(Comment::where(['product_id' => $product->id, 'user_id' => Auth::id(), 'id' => $request->delete_comment_id])->delete())
        {
          Comment::where(['product_id' => $product->id, 'parent' => $request->delete_comment_id])->delete();
        }

        return redirect(url()->current().'?tab=comments');
      }
 
      if($request->isMethod('POST'))
      {
        $type = $request->input('type');
        
        $redirect_url = $request->redirect_url ?? $request->server('HTTP_REFERER');

        if($type === 'reviews')
        {
          config('app.enable_reviews') ?? abort(404);

          if(!$product->purchased && !$product->valid_subscription) abort(404);

          $rating  = $request->input('rating');
          $review  = $request->input('review');
          $approved = auth_is_admin() ? 1 : (config('app.auto_approve.reviews') ? 1 : 0);

          if(!filter_var($rating, FILTER_VALIDATE_INT)) return redirect($redirect_url);

          if($request->post('edit_review_id'))
          {
            $_review = Review::where(['product_id' => $product->id, 'user_id' => Auth::id(), 'id' => $request->edit_review_id])->first();

            $_review->content = $request->post('review');
            $_review->rating = $request->post('rating');
            $_review->updated_at = date('Y-m-d H:i:s');
            $_review->approved = $approved;

            $_review->save();

            $redirect_url = ur()->current()."?tab=reviews#rev-{$_review->id}";
          }
          else
          {
            DB::insert("INSERT INTO reviews (product_id, user_id, rating, content, approved) VALUES (?, ?, ?, ?, ?) 
                      ON DUPLICATE KEY UPDATE rating = ?, content = ?", 
                      [$product->id, $user_id, $rating, $review, $approved, $rating, $review]);

            if(!$approved)
            {
              $request->session()->put(['review_response' => 'Your review is waiting for approval. Thank you!']);
            }

            if(Auth::check() && !auth_is_admin() && config('app.admin_notifications.reviews'))
            {
                $mail_props = [
                  'data'   => ['text' => __('A new review has been posted by :user for :item.', ['user' => $request->user()->name ?? null, 'item' => $product->name]),
                               'subject' => __('You have a new review.'),
                               'user_email' => $request->user()->email],
                  'action' => 'send',
                  'view'   => 'mail.message',
                  'reply_to' => $request->user()->email,
                  'to'     => config('app.email'),
                  'subject' => __('You have a new review.')
                ];

                sendEmailMessage($mail_props, config('mail.mailers.smtp.use_queue'));
            }  
          }
          
          $ratings = Review::select('rating')->where('product_id', $product->id)->get()->pluck('rating')->toArray();
          
          $reviews = count($ratings);
          $rating  = ceil(array_sum($ratings) / count($ratings)+1);

          DB::update("UPDATE products SET rating = $rating, reviews = ? WHERE products.id = ?", [$reviews, $product->id]);

        }
        elseif(preg_match('/^support|comments$/i', $type))
        {
          config('app.enable_comments') ?? abort(404);

          if(!$comment = $request->input('comment'))
          {
            return redirect($redirect_url);
          }

          $approved = auth_is_admin() ? 1 : (config('app.auto_approve.support') ? 1 : 0);
          $comment  = strip_tags($comment);

          if($request->edit_comment_id)
          {
            $edit_comment = Comment::where('id', $request->edit_comment_id)->where('user_id', Auth::id())
                            ->where('product_id', $product->id)->first();

            if($edit_comment)
            {
              $edit_comment->body       = $comment;
              $edit_comment->approved   = $approved;
              $edit_comment->updated_at = date("Y-m-d H:i:s");

              $edit_comment->save();
            }
          }
          else
          {
            if($request->comment_id) // parent
            {
              if($parent_comment = Comment::where('id', $request->comment_id)->where('parent', null)->where('product_id', $product->id)->first())
              {
                DB::insert("INSERT INTO comments (product_id, user_id, body, approved, parent) VALUES (?, ?, ?, ?, ?)", 
                            [$product->id, $user_id, $comment, $approved, $parent_comment->id]);    
              }
            }
            else
            {
              DB::insert("INSERT INTO comments (product_id, user_id, body, approved) VALUES (?, ?, ?, ?)", 
                          [$product->id, $user_id, $comment, $approved]); 
            }

            if(!$approved)
            {
              $request->session()->put(['comment_response' => __('Your comment is waiting for approval. Thank you!')]);
            }

            if(Auth::check() && !auth_is_admin() && config('app.admin_notifications.comments'))
            {
                $mail_props = [
                  'data'   => ['text' => __('A new comment has been posted by :user for :item.', ['user' => $request->user()->name ?? null, 'item' => $product->name]),
                               'subject' => __('You have a new comment.'),
                               'user_email' => $request->user()->email
                             ],
                  'action' => 'send',
                  'view'   => 'mail.message',
                  'to'     => config('app.email'),
                  'subject' => __('You have a new comment.')
                ];

                sendEmailMessage($mail_props, config('mail.mailers.smtp.use_queue'));
            }
          }
        }

        return redirect($redirect_url);
      }

      $custom_meta_tags = json_decode($product->meta_tags ?? '');

      config([
        "meta_data.title" => $custom_meta_tags->title ?? $product->name,
        "meta_data.description" => $custom_meta_tags->description ?? $product->short_description,
        "meta_data.image" => asset("storage/covers/{$product->cover}"),
        "meta_tags.keywords" => $custom_meta_tags->keywords ?? $product->tags,
        "json_ld" => $json_ld
      ]);

      DB::update('UPDATE products USE INDEX(primary) SET views = views+1 WHERE id = ?', [$product->id]);

      if(auth_is_admin() || $product->purchased || $product->valid_subscription)
      {
          $temp_direct_url = Temp_Direct_Url::where('product_id', $product->id)->first();

          if(!$temp_direct_url)
          {
            (new \App\Http\Controllers\ProductsController)->update_temp_direc_url($product);

            $temp_direct_url = Temp_Direct_Url::where('product_id', $product->id)->first();
          }

          $product->setAttribute('temp_direct_url', $temp_direct_url ? $temp_direct_url->url : ($product->direct_download_link ?? null));
      }

      if(!$product->for_subscriptions)
      {
          $product->fill(['has_keys' => $product->has_keys()]);
      }

      $data['product']->category->increment('views', 1);
    
      if(!$product->comments_count && is_array($product->fake_comments) && count($product->fake_comments ?? []) > 0)
      {
          $data['product']->comments_count = count($product->fake_comments ?? []);
      }

      return view_('product', $data);
    }




    public function product_with_permalink(Request $request)
    {
      $request->merge(['via_permalink' => 1]);

      return $this->product($request);
    }



    // Redirect old product URLs to new URLs
    public function old_product_redirect(Request $request)
    {
      $product = Product::where(['slug' => $request->slug, 'active' => 1])->first() ?? abort(404);

      return redirect(item_url($product));
    }



    public function prepaid_credits(Request $request)
    {
      $packs = Prepaid_Credit::orderBy('order', 'asc')->get();

      config([
        "meta_data.title" => __('Deposit cash'),
        "meta_data.description" => __('Add prepaid credits'),
        'json_ld' => [
          '@context' => 'http://schema.org',
          '@type' => 'WebPage',
          'name' => __('Prepaid credits'),
          'description' => __('Prepaid credits'),
          'url' => route('home.prepaid_credits'),
        ],
      ]);

      foreach($packs as &$pack)
      {
          $pack->specs = array_filter(explode("\r\n", $pack->specs));
      } 

      return view_('prepaid_credits', compact('packs'));
    }


    // Featured products
    private static function _featured_products(bool $returnQueryBuilder, $limit = 15, $category_id = null, $randomize = false)
    {
      $db_columns = [
        'products.id',
        'products.name',
        'products.slug',
        'products.price',
        'products.promo_price',
        'products.promo_time',
        DB::raw('(products.promo_time IS NOT NULL and products.promo_time > UNIX_TIMESTAMP()) as has_promo'),
        'products.short_description',
        'products.sales',
        'products.rating',
        'products.category as category_id',
        'products.release_date',
        'categories.name as category_name',
        'categories.slug as category_name',
      ];

      $products = Product::forceIndex('category')
                  ->select($db_columns)
                  ->leftJoin('categories', 'categories.id', '=', 'products.category')
                  ->where(['products.featured' => 1, 'products.active' => 1]);

      $products = $category_id ? $products->where('category', $category_id) : $products;

      $products = $randomize ? $products->orderByRaw('RAND()') : $products;
      
      return $returnQueryBuilder ? $products : $products->limit($limit)->get();   
    }



    // Blog 
    public function blog(Request $request)
    {      
      $filter = [];

      config([
        "meta_data.title" => config('app.blog.title'),
        "meta_data.description" => config('app.blog.description'),
        "meta_data.image" => asset('storage/images/'.(config('app.blog_cover') ?? 'blog_cover.jpg')),
      ]);

      if($request->category)
      {
        if(!$category = Category::useIndex('slug')->where('slug', $request->category)->first())
          abort(404);

        $posts = Post::useIndex('category')->where(['category' => $category->id, 'active' => 1]);

        $filter = ['name' => 'Category', 'value' => $category->name];

        config([
          "meta_data.title" => config('app.name').' Blog - '.$category->name,
          "meta_data.description" => $category->description,
        ]);
      }
      elseif($request->tag)
      {
        $posts = Post::useIndex('tags')->where(function ($query) use ($request) {
                                                        $tag = str_replace('-', ' ', $request->tag);

                                                        $query->where('tags', 'LIKE', "%{$request->tag}%")
                                                              ->orWhere('tags', 'like', "%{$tag}%");
                                                   })
                                                   ->where('active', 1);

        $filter = ['name' => 'Tag', 'value' => $request->tag];

        config([
          "meta_data.title" => config('app.name').' Blog - '.$request->tag,
        ]);
      }
      elseif($request->q)
      {
        $request->tag = str_replace('-', ' ', $request->tag);
        $posts = Post::useIndex('search', 'active')->where(function ($query) use ($request) {
                                                         $query->where('posts.name', 'like', "%{$request->q}%")
                                                               ->orWhere('posts.tags', 'like', "%{$request->q}%")
                                                               ->orWhere('posts.slug', 'like', "%{$request->q}%");
                                                     })
                                                     ->where('active', 1);

        $filter = ['name' => 'Search', 'value' => $request->q];

        config([
          "meta_data.title" => config('app.name').' '.__('Blog').' - '.__('Searching for').' '.$request->q,
        ]);
      }
      else
      {
        $posts = Post::useIndex('primary')->where('active', 1);
      }

      $posts = $posts->select('posts.*', 'categories.name as category_name', 'categories.slug as category_slug')
                      ->leftJoin('categories', 'categories.id', '=', 'posts.category')
                      ->where('posts.active', 1)->orderBy('id', 'desc')->paginate(8);

      foreach($posts->items() as $post)
      {
        $post->tags = array_slice(array_filter(array_map('trim', explode(',', $post->tags))), 0, 3);
      }

      if($filter) settype($filter, 'object');

      $posts_categories = Category::useIndex('`for`')->select('name', 'slug')->where('categories.for', 0)->get();

      $latest_posts = Post::useIndex('primary', 'active')
                      ->select('posts.*', 'categories.name as category_name', 'categories.slug as category_slug')
                      ->leftJoin('categories', 'categories.id', '=', 'posts.category')
                      ->where('posts.active', 1)->orderBy('updated_at')->limit(4)->get();

      foreach($latest_posts as $latest_post)
      {
        $latest_post->tags = array_slice(array_filter(array_map('trim', explode(',', $latest_post->tags))), 0, 3);
      }


      $posts_tags = Post::useIndex('active')->select('tags')->where('active', 1)->orderByRaw('rand()')
                                  ->limit(10)->get()->pluck('tags')->toArray();

      $tags  = [];

      foreach($posts_tags as $tag)
      {
        $tags = array_merge($tags, array_map('trim', explode(',', $tag)));
      }

      $tags = array_unique($tags);

      config([
        'json_ld' => [
          '@context' => 'http://schema.org',
          '@type' => 'Blog',
          '@id' => route('home.blog'),
          'name' => __('Blog'),
          'url' => route('home.blog'),
          'image' => asset("storage/images/" . config('blog_cover')),
          'description' => config('app.blog.description'),
        ]
      ]);

      return view_('blog', compact('posts_categories', 'latest_posts', 'tags', 'posts', 'filter'));
    }


    // BLOG POST
    public function post(string $slug)
    {
      $post = Post::useIndex('slug', 'active')->select('posts.*', 'categories.name AS category', 'posts.category as category_id')
                  ->leftJoin('categories', 'categories.id', '=', 'posts.category')
                  ->where(['posts.slug' => $slug, 'posts.active' => 1])->first() ?? abort(404);

      config([
        "meta_data.title" => $post->name,
        "meta_data.description" => $post->short_description,
        "meta_data.image" => asset('storage/posts/'.$post->cover)
      ]);

      $post->setTable('posts')->increment('views', 1);

      $latest_posts = Post::useIndex('primary', 'active')
                      ->select('posts.*', 'categories.name as category_name', 'categories.slug as category_slug')
                      ->leftJoin('categories', 'categories.id', '=', 'posts.category')
                      ->where('posts.id', '!=', $post->id)->where('posts.active', 1)->orderBy('updated_at')->limit(5)->get();

      $related_posts =  Post::useIndex('primary', 'active')
                        ->select('posts.*', 'categories.name as category_name', 'categories.slug as category_slug')
                        ->leftJoin('categories', 'categories.id', '=', 'posts.category')
                        ->where('posts.id', '!=', $post->id)->where('posts.active', 1)
                        ->where('posts.category', $post->category_id)->orderBy('updated_at')->limit(6)->get();


      foreach($related_posts as $related_post)
      {
        $related_post->tags = array_slice(array_filter(array_map('trim', explode(',', $related_post->tags))), 0, 3);
      }

      $posts_categories = Category::useIndex('`for`')->select('name', 'slug')->where('categories.for', 0)->get();
      $posts_tags       = Post::useIndex('active')->select('tags')->where('active', 1)->orderByRaw('rand()')
                                  ->limit(10)->get()->pluck('tags')->toArray();

      $tags  = [];

      foreach($posts_tags as $tag)
      {
        $tags = array_merge($tags, array_map('trim', explode(',', $tag)));
      }

      $tags = array_unique($tags);

      config([
        'json_ld' => [
          '@context' => 'https://schema.org/',
          '@type' => 'BlogPosting',
          '@id' => url()->current(),
          'mainEntityOfPage' => url()->current(),
          'headline' => $post->name,
          'name' => $post->name,
          'description' => $post->short_description,
          'datePublished' => $post->created_at,
          'dateModified' => $post->updated_at,
          'image' => [
            '@type' => 'ImageObject',
            '@id' => asset("storage/images/{$post->cover}"),
            'url' => asset("storage/images/{$post->cover}"),
            'height' => 500,
            'width' => 500,
          ],
          'url' => url()->current(),
          'wordCount' => str_word_count(strip_tags($post->content)),
          'keywords' => array_filter(explode(',', $post->keywords)),
        ]
      ]);

      $post_link = route('home.post', ['slug' => $post->slug]);

      return view_('post', compact('post', 'posts_categories', 'latest_posts', 'related_posts', 'tags', 'post_link'));
    }


    // Send email verification link
    public function send_email_verification_link(Request $request)
    {
      $notifiable = $request->email ? User::where('email', $request->email)->first() : $request->user();
      $notifiable->sendEmailVerificationNotification();

      return response()->json([
        'status' => true, 
        'message' => __('Please check your :email inbox for a verification link.', ['email' => $request->email])
      ]);
    }


    private function get_random_product()
    {
      return  Product::selectRaw('products.name as item_name, products.slug, products.cover, products.id, products.price, products.rating, products.fake_rating as rating_2')
              ->where('products.active', 1)
              ->where('products.price', '>', 0)
              ->orderByRaw('RAND()')
              ->first();
    }


    public function live_sales(Request $request)
    {
      $response = [];

      $i_min = config('app.fake_purchases.interval.min', 10);
      $i_max = config('app.fake_purchases.interval.max', 20);

      $fake_profiles = shuffle_array(config('app.fake_profiles', []));

      $response = ['status' => 0, 'sale' => []];

      $product = $this->get_random_product();

      if($product && $fake_profiles)
      {
        $response['status'] = 1;
        $product->price = price($product->price, false, true, 2, 'code');
        $product->cover = src("storage/covers/{$product->cover}");
        $product->url   = item_url($product);

        $html_rating = '';

        foreach(rating($product->fake_rating ?? $product->rating) as $rating_img)
        {
            $html_rating .= <<<HTML
            <img src="/assets/images/{$rating_img}" class="w-[20px] h-[20px]">
            HTML;
        }

        $product->setAttribute('html_rating', $html_rating);

        $response['sale'] = $product->toArray();
        $fake_profile = $fake_profiles[rand(0, (count($fake_profiles) - 1))];
        $response['sale'] = array_merge($response['sale'], $fake_profile);
      }

      return json($response);
    }



    public function invoice(Request $request)
    {
      $required_params = ["buyer_name", "buyer_email", "date", "reference", "items", "subtotal", "fee", "tax", 
                        "discount", "total_due", "refunded", "is_subscription", "currency"];

      $missing_params = array_diff($required_params, array_keys($_GET));

      if(count($missing_params))
      {
          abort(403, __("The folowing parameters are missing (:params)", ['params' => implode(', ', $missing_params)]));
      }

      $content  = base64_encode(file_get_contents(public_path("storage/images/".config('app.logo'))));
      $mimetype = mime_content_type(public_path("storage/images/".config('app.logo')));

      config(["logo_b64" => "data:{$mimetype};base64,{$content}"]);

      return view('invoices.template_2', $_GET);
    }



    public function export_invoice(Request $request)
    {
        $transaction_id = $request->itemId ?? abort(404);
        $transaction = Transaction::find($transaction_id) ?? abort(404);
        $buyer = User::find($transaction->user_id) ?? abort(404);

        if(!$details = json_decode($transaction->details, true))
        {
            return back();
        }

        $items = array_filter($details['cart'] ?? $details['items'], function($k)
        {
            return is_numeric($k);
        }, ARRAY_FILTER_USE_KEY);

        $fee             = $details['fee'] ?? 0;
        $tax             = $details['tax'] ?? 0;
        $discount        = $details['discount'] ?? 0;
        $subtotal        = array_sum(array_column($items, 'price'));
        $total_due       = $details['total_amount'] ?? $details['amount'];
        $currency        = $details['currency'];
        $refunded        = $transaction->refunded;
        $reference       = $transaction->reference_id ?? $transaction->order_id ?? $transaction->transaction_id;
        $is_subscription = $transaction->is_subscription;
        $custom_amount   = $transaction->custom_amount;

        $data = compact('items', 'fee', 'tax', 'discount', 'subtotal', 'currency', 'is_subscription',
                        'total_due', 'reference', 'transaction', 'buyer', 'refunded', 'custom_amount');

        $invoice_data = http_build_query([
            'items' => $items,
            'fee' => $fee,
            'tax' => $tax,
            'discount' => $discount,
            'subtotal' => $subtotal,
            'currency' => $currency,
            'is_subscription' => $is_subscription,
            'total_due' => $total_due,
            'reference' => $reference,
            'date' => $transaction->created_at->format('Y-m-d'),
            'buyer_name' => $buyer->name,
            'buyer_email' => $buyer->email,
            'refunded' => $refunded,
            'custom_amount' => $custom_amount
        ]);

        if(config('app.invoice.template') == 2)
        {
            return json(['url' => route('home.invoice', $invoice_data)]);
        }

        $pdf = \PDF::loadView('invoices.template_1', compact('items', 'fee', 'tax', 'discount', 'subtotal', 'currency', 'is_subscription',
                                                'total_due', 'reference', 'transaction', 'buyer', 'refunded', 'custom_amount'));
        return $pdf->download('invoice.pdf'); // stream | download
    }



    // Download
    public function download(Request $request)
    {
      set_time_limit(1800); // 30 mins

      $order_id = is_numeric($request->order_id) ? $request->order_id : abort(404);
      $user_id  = (Str::isUuid($request->user_id) || is_numeric($request->user_id)) ? $request->user_id : abort(404);
      $item_id  = is_numeric($request->item_id) ? $request->item_id : abort(404);
      $type     = strtolower($request->type);

      if(!in_array($type, ['file', 'license', 'key']))
      {
        abort(403, __('Wrong request type.')); 
      }

      $transaction =  Transaction::where(['id' => $order_id])->where(function($query) use($user_id)
                      {
                        $query->where('user_id', $user_id)
                              ->orWhere('guest_token', $user_id);
                      })->first();

      $item = Product::where(['active' => 1, 'id' => $item_id])->first() ?? abort(404);

      if($type === 'file' && (!$item->file_name && !$item->direct_download_link))
      {
        abort(403, __('No file was uploaded for this item yet. Please contact support at :email', ['email' => config('app.email')]));
      }

      $item->free = json_decode($item->free, true);

      $item->setAttribute('can_download', auth_is_admin() ? 1 : 0);

      if($item->price === 0)
      {
        if(!Auth::check() && config('app.authentication_required_to_download_free_items') && !Str::isUuid($request->user_id))
        {
          abort(403, __('You must be logged in to download free items'));
        }

        $item->can_download = 1;
      }

      if(!$item->can_download)
      {
        if(\Auth::check() && (Auth::id() != $user_id))
        {
          abort(403, __("You are not allowed to download this file"));
        }

        $transaction ?? abort(404);

        if($transaction->status != "paid" || $transaction->confirmed != 1)
        {
          abort(403, __('The payment for your order :number is not confirmed yet, please try again later.', ['number' => $transaction->reference_id]));
        }

        if($transaction->sandbox == 1 && !config('app.allow_download_in_test_mode'))
        {
          abort(403, __('Download not allowed with test orders'));
        }

        if($transaction->refunded == 1)
        {
          abort(403, __('This Order :number has been refunded', ['number' => $transaction->reference_id]));
        }

        if($transaction->type == "product")
        {
          $products_ids = array_map('trim', explode(',', str_ireplace("'", "", $transaction->products_ids)));
          
          in_array($item_id, $products_ids) ?? abort(404);

          $item->can_download = 1;
        }
        elseif($transaction->type == "subscription")
        {
          $item->can_download = 1;

          $subscription = User_Subscription::where('transaction_id', $order_id)->first() ?? abort(404);

          $ends_at_time   = strtotime($subscription->ends_at);
          $now_time       = strtotime("now");

          if($ends_at_time < $now_time)
          {
            abort(403, __('Subscription expired'));
          }

          $pricing_table       = Pricing_Table::where('id', $subscription->subscription_id)->first() ?? abort(404);
          $same_item_downloads = Subscription_Same_Item_Downloads::where(['subscription_id' => $subscription->id, 'product_id' => $item_id])->first();

          if($pricing_table->limit_downloads_same_item > 0)
          {
            if(!$same_item_downloads)
            {
              $same_item_downloads = Subscription_Same_Item_Downloads::create([
                'subscription_id' => $subscription->id,
                'product_id' => $item_id,
                'downloads' => 0
              ]);
            }

            if($same_item_downloads->downloads == $pricing_table->limit_downloads_same_item)
            {
              abort(403, __('Download limit reached'));
            }
          }

          if($pricing_table->limit_downloads_per_day > 0)
          {
            if(is_null($subscription->daily_downloads_date))
            {
              $subscription->daily_downloads_date = date('Y-m-d');
              $subscription->daily_downloads += 1; 

              $subscription->save();
            }
            else
            {
              if(strtotime($subscription->daily_downloads_date) < strtotime(date('Y-m-d')))
              {
                $subscription->daily_downloads_date = date('Y-m-d');
                $subscription->daily_downloads = 0;

                $subscription->save();
              }
              elseif($subscription->daily_downloads_date == date("Y-m-d"))
              {
                if($subscription->daily_downloads == $pricing_table->limit_downloads_per_day)
                {
                  abort(403, __('Download limit reached for today'));
                }
                else
                {
                  $subscription->daily_downloads += 1; 

                  $subscription->save();
                }
              }
            }
          }
          
          if($pricing_table->categories)
          {
            $categories = array_filter(explode(',', $pricing_table->categories));
            $categories = filter_var_array($categories, FILTER_VALIDATE_INT);

            // Requested item belongs to valid subscription category
            if(!$valid_subscription_category = Product::select('id')->whereIn('products.category', $categories)->where('products.id', $item_id)->exists())
            {
              abort(403, __('You are not allowed to download this item'));
            }
          }
        }
      }

      if($item->can_download)
      {
        if($type == "file")
        {
          if($item->direct_download_link)
          {
              return redirect()->away($item->direct_download_link);
          }
          
          if($item->file_host == 'local')
          {          
            if(file_exists(storage_path("app/downloads/{$item->file_name}")))
            {
              return response()->streamDownload(function() use($item)
              {
                readfile(storage_path("app/downloads/{$item->file_name}"));
              }, "{$item->slug}.{$item->file_extension}", ["Content-Type" => "application/octet-stream"]);
            }
          }
          else
          {
            $host_class = [
              'dropbox'   => 'DropBox',
              'google'    => 'GoogleDrive',
              'yandex'    => 'YandexDisk',
              'amazon_s3' => 'AmazonS3',
              'wasabi'    => 'Wasabi',
              'gcs'       => "GoogleCloudStorage",
            ];

            $class_name = $host_class[$item->file_host];

            try
            {
              $config = [
                "item_id"    => $item->file_name,
                "cache_id"   => $item->id,
                "file_name"  => "{$item->id}-{$item->slug}.{$item->file_extension}",
                "expiry"     => null,
                "bucketName" => null,
                "bucket"     => null,
                "options"    => null,
              ];

              $response = call_user_func(["\App\Libraries\\{$class_name}", 'download'], $config);

              if(config('app.force_download'))
              {
                header("Content-disposition:attachment; filename={$config['file_name']}");
                readfile($response->getTargetUrl());
                exit;
              }

              return $response;
            }
            catch(\Exception $e)
            {
              abort(403, __('Could not download the main file. Please contact support at :email', ['email' => config('app.email')]));
            }
          }
        }
        elseif($type == "license")
        {
          try 
          {
            $license = json_decode(decrypt(urldecode($request->query('content')), false));

            $user_email = $transaction->user_id ? User::find($transaction->user_id)->email : $transaction->guest_email;

            $replaces = [
              "{LICENSE_NAME}"  => $license->name,
              "{APP_NAME}"      => config('app.name'),
              "{APP_OWNER}"     => config('app.email'),
              "{BUYER_EMAIL_ADDRESS}" => $user_email,
              "{ITEM_NAME}"     => $item->name,
              "{ITEM_ID}"       => $item->id,
              "{ITEM_URL}"      => item_url($item),
              "{LICENSE_KEY}"   => $license->license,
              "{PURCHASE_DATE}" => $transaction->created_at,
              "{CONTACT_PAGE_URL}" => route('home.support')
            ];

            $license_model = file_get_contents(resource_path("extra/license_model.txt"));
            $license_file  = str_ireplace(array_keys($replaces), array_values($replaces), $license_model);

            return response()->streamDownload(function() use($license_file)
            {
              echo $license_file;
            }, "{$item->id}-{$item->slug}-license.txt")->send();
          }
          catch(\Exception $e)
          {
            abort(403, __('Could not download the license key. Please contact support at :email', ['email' => config('app.email')]));
          }
        }
        elseif($type == "key")
        {
          try
          {
            $key = decrypt(urldecode($request->query('content')), false);

            return response()->streamDownload(function() use($key)
            {
              echo $key;
            }, "{$item->id}-{$item->slug}-key.txt")->send();
          }
          catch(\Exception $e)
          {
            abort(403, __('Could not download the key. Please contact support at :email', ['email' => config('app.email')]));
          }
        }
      }

      return redirect('/');
    }




    // Support
    public function support(Request $request)
    {
      if($request->method() === 'POST')
      {
        $rules = [
          'email' => 'required|email|bail',
          'subject' => 'required|bail',
          'message' => 'required'
        ];

        if(captcha_is_enabled('contact'))
        {
          if(captcha_is('mewebstudio'))
          {
            $rules['captcha'] = 'required|captcha';
          }
          elseif(captcha_is('google'))
          {
            $rules['g-recaptcha-response'] = 'required';
          }
        }

        $request->validate($rules, [
            'g-recaptcha-response.required' => __('Please verify that you are not a robot.'),
            'captcha.required' => __('Please verify that you are not a robot.'),
            'captcha.captcha' => __('Wrong captcha, please try again.'),
        ]);

        $user_email = $request->input('email');

        $email = Support_Email::insertOrIgnore(['email' => $user_email]);

        if(!($email->id ?? null))
        {
          $email = Support_Email::where('email', $user_email)->first();
        }

        $support = new Support();

        $support->email_id = $email->id;
        $support->subject  = strip_tags($request->input('subject'));
        $support->message  = strip_tags($request->input('message'));

        $support->save();

        $mail_props = [
          'data'   => ['subject' => $support->subject, 'text' => $support->message, 'user_email' => $user_email],
          'action' => 'send',
          'view'   => 'mail.message',
          'to'     => config('app.email'),
          'subject' => $support->subject,
          'reply_to' => $user_email,
          'forward_to' => config('mail.forward_to')
        ];

        sendEmailMessage($mail_props, config('mail.mailers.smtp.use_queue'));

        $request->session()->flash('support_response', __('Message sent successfully'));

        return redirect()->route('home.support', ['#contact-form']);
      }

      $faqs = Faq::useIndex('active')->where('active', 1)->get();

      config([
        "meta_data.title" => __('Support'),
        "meta_data.description" => __('Support'),
        "json_ld" => [
          '@context' => 'http://schema.org',
          '@type' => 'WebPage',
          'name' => __('Support and FAQ'),
          'description' => __('Support and FAQ'),
          'url' => url()->current()
        ]
      ]);

      $support = Page::where('slug', 'support')->first();
      
      return view_('support', compact('faqs', 'support'));
    }



    // Pricing
    public function subscriptions(Request $request)
    {
      $subscriptions = Pricing_Table::useIndex('position')->orderBy('position', 'asc')->get();

      $active_subscription = null;

      config([
        "meta_data.title" => __('Pricing - :app_name', ['app_name' => config('app.name')]),
        "meta_data.description" => __('Pricing - :app_name', ['app_name' => config('app.name')]),
        "json_ld" => [
          '@context' => 'http://schema.org',
          '@type' => 'WebPage',
          'name' => __('Pricing'),
          'description' => __('Pricing plans'),
          'url' => url()->current()
        ]
      ]);

      if(Auth::check() && !config('app.subscriptions.accumulative'))
      {
        $user_subscription =  User_Subscription::useIndex('user_id', 'subscription_id')
                              ->select('user_subscription.id')
                              ->join(DB::raw('pricing_table USE INDEX(primary)'), 'pricing_table.id', '=', 'user_subscription.subscription_id')
                              ->join(DB::raw('transactions USE INDEX(products_ids, is_subscription)'), function($join)
                              {
                                $join->on('transactions.products_ids', '=', DB::raw('QUOTE(pricing_table.id)'))
                                     ->where('transactions.is_subscription', '=', 1);
                              })
                              ->where('user_subscription.user_id', Auth::id())
                              ->whereRaw("user_subscription.ends_at IS NOT NULL AND CURRENT_TIMESTAMP < user_subscription.ends_at")
                              ->where(function($query)
                              {
                                $query->where('transactions.refunded', '0')
                                      ->orWhere('transactions.refunded', null);
                              })
                              ->first();

        $active_subscription = $user_subscription ? true : false;
      }

      foreach($subscriptions as &$subscription)
      {
        $subscription->specifications = json_decode($subscription->specifications); 
      }

      return view_('pricing', compact('subscriptions', 'active_subscription'));
    }



    // Checkout
    public function checkout(Request $request)
    {
      config([
          "meta_data.title" => __('Checkout'),
      ]);

      $type = strtolower($request->query('type')) ?? abort(404);
      $coupon_code = $request->query('coupon', '');
      $payment_gateway = $request->query('gateway');

      if($payment_gateway && !config("payments.gateways.{$payment_gateway}.enabled"))
      {
          abort(404);
      }

      in_array($type, ['subscription', 'prepaid-credits', 'items']) ?? abort(404);

      $payment_processors = array_filter(config('payments.gateways'), function($gateway)
      {
          return isset($gateway['enabled']);
      });

      if(config("payments.gateways.{$payment_gateway}.enabled"))
      {
          $selected_gateway = $payment_processors[$payment_gateway];

          unset($payment_processors[$payment_gateway]);

          $payment_processors = array_merge([$payment_gateway => $selected_gateway], $payment_processors);
      }

      $payment_processor = count($payment_processors) > 1 ? null : $payment_processors->first();

      $subscription   = null;
      $prepaid_credit = null;
      $cart_items     = [];
      $total_amount   = 0;
      $due_amount     = 0;
      $coupon         = null;
      $discount       = 0;
      $tax            = 0;
      $fee            = 0;

      if($type === 'subscription')
      { 
          $subscription = Pricing_Table::find($request->id) ?? abort(404);

          if(!config('app.subscriptions.accumulative'))
          {
              $active_subscription =  User_Subscription::useIndex('user_id', 'subscription_id')
                                  ->select('user_subscription.id')
                                  ->join(DB::raw('pricing_table USE INDEX(primary)'), 'pricing_table.id', '=', 'user_subscription.subscription_id')
                                  ->join(DB::raw('transactions USE INDEX(products_ids, is_subscription)'), function($join)
                                  {
                                    $join->on('transactions.products_ids', '=', DB::raw('QUOTE(pricing_table.id)'))
                                         ->where('transactions.is_subscription', '=', 1);
                                  })
                                  ->where('user_subscription.user_id', Auth::id())
                                  ->whereRaw("user_subscription.ends_at IS NOT NULL AND CURRENT_TIMESTAMP < user_subscription.ends_at")
                                  ->where(function($query)
                                  {
                                    $query->where('transactions.refunded', '0')
                                          ->orWhere('transactions.refunded', null);
                                  })
                                  ->first();

              if($active_subscription ?? false)
              {
                return redirect('/')->with(['user_message' => __("It's not possible to subscribe to another membership plan while your previous one has not expired yet.")]);
              }
          }

          $subscription->price = price($subscription->price, false, false, 0, null, null);

          $subscription->specifications = json_decode($subscription->specifications);

          $total_amount = $subscription->price;

          $due_amount = $total_amount;

          if(trim($coupon_code))
          {
              if($coupon = Coupon::where('code', $coupon_code)->where('for', 'subscriptions')->first())
              {
                  if($coupon->is_percentage)
                  {
                      $discount   = $total_amount * ($coupon->value/100);

                      $due_amount = ($total_amount - $discount) > 0 ? ($total_amount - $discount) : 0;
                  }
                  else
                  {
                      $discount   = ($total_amount - $coupon->value) > 0 ? $coupon->value : $total_amount;

                      $due_amount = ($total_amount - $coupon->value) > 0 ? ($total_amount - $coupon->value) : 0;
                  }
              }
          }
      }
      elseif($type === 'prepaid-credits')
      {
          $prepaid_credit = Prepaid_Credit::find($request->id)  ?? abort(404);

          $prepaid_credit->specs = array_filter(explode("\r\n", $prepaid_credit->specs));

          $total_amount = $due_amount = $prepaid_credit->amount;
      }
      elseif($type === "items")
      {
          $uuid = $request->query('id') ?? abort(404);

          $cart_items = User_Shopping_Cart_Item::where('uuid', $uuid)->get()->toArray();

          $cart_items = array_combine(array_column($cart_items, 'id'), $cart_items);

          foreach($cart_items as $cart_item)
          {
              $total_amount += $cart_item['price'];
          }

          $due_amount = $total_amount;

          if(trim($coupon_code))
          {
              if($coupon = Coupon::where('code', $coupon_code)->where('for', 'products')->first())
              {
                  if($coupon->is_percentage)
                  {
                      $discount   = $total_amount * ($coupon->value/100);

                      $due_amount = ($total_amount - $discount) > 0 ? ($total_amount - $discount) : 0;
                  }
                  else
                  {
                      $discount   = ($total_amount - $coupon->value) > 0 ? $coupon->value : $total_amount;

                      $due_amount = ($total_amount - $coupon->value) > 0 ? ($total_amount - $coupon->value) : 0;
                  }
              }
          }
      }
      else
      {
          return redirect()->route('home');
      }

      if(!count($cart_items) && !$subscription && !$prepaid_credit)
      {
          return redirect()->route('home');
      }

      if($due_amount > 0)
      {
          if(config("payments.gateways.{$payment_gateway}.fee") && $type !== 'prepaid-credits')
          {
              $fee = format_amount(config("payments.gateways.{$payment_gateway}.fee"));

              $due_amount += $fee;   
          }

          if(config('payments.vat') && $type !== 'prepaid-credits')
          {
              $tax = format_amount($due_amount * config('payments.vat') / 100);

              $due_amount += $tax;
          }
      }

      if($payment_gateway || $due_amount <= 0)
      {
          $checkout_url = route('home.checkout', []);
      }

      if($payment_gateway === 'credits' && $type === 'prepaid-credits')
      {
          $url_query = parse_url(url()->full(), PHP_URL_QUERY);
          
          parse_str($url_query, $query_arr);

          unset($query_arr['gateway']);

          return redirect()->route('home.checkout', $query_arr)->with(['user_message' => __('You cannot pay a prepaid credits pack with your credits')]);
      }

      $checkout_params = [
          'payment_processors' => $payment_processors,
          'subscription'       => $subscription,
          'prepaid_credit'     => $prepaid_credit,
          'cart_items'         => $cart_items,
          'total_amount'       => $total_amount,
          'due_amount'         => $due_amount,
          'discount'           => $discount,
          'coupon'             => $coupon,
          'payment_gateway'    => config("payments.gateways.{$payment_gateway}"),
          'tax'                => $tax,
          'fee'                => $fee,
          'type'               => $type, // subscription or prepaid_credit or items
      ];


      $checkout_config = [
          'total_amount'       => $total_amount,
          'due_amount'         => $due_amount,
          'discount'           => $discount,
          'coupon'             => $coupon->code ?? null,
          'payment_gateway'    => config("payments.gateways.{$payment_gateway}.name"),
          'tax'                => $tax,
          'fee'                => $fee,
          'type'               => $type,
      ];
        
      $view_data = array_merge(['title' => __(':app_name - Checkout', ['app_name' => config('app.name')])], $checkout_params, ['checkout_config' => $checkout_config, 'checkout_params' => $checkout_params]);

      $view_data['payment_gateway']['name'] = $view_data['payment_gateway']['name'] ?? null;

      return view_('checkout', $view_data);
    }


    public function checkout_error(Request $request)
    {
      $message = session('message') ?? abort(404);

      config([
        "meta_data.title" => __('Payment failed'),
      ]);

      return view_('checkout.failure', compact('message',));
    }



    // List Product folder To Preview Its Files (POST)
    public function product_folder_async(Request $request)
    {
      $request->validate([
        'id' => 'required|numeric',
        'slug' => 'required|string'
      ]);

      config('filehosts.working_with') == 'folders' || abort(404);

      $item = Product::useIndex('primary')->select('file_name', 'file_host')
                                          ->where(['slug' => $request->slug, 'id' => $request->id])->first() ?? abort(404);

      if($item->file_host === 'google')
      {
        $files_list = GoogleDrive::list_folder($item->file_name)->original['files_list'] ?? [];
      }
      /*if($item->file_host === 'onedrive')
      {
        $files_list = OneDrive::list_folder($item->file_name)->original['files_list'] ?? [];
      }*/
      elseif($item->file_host === 'dropbox')
      {
        $files_list = DropBox::list_folder($item->file_name)->original['files_list'] ?? [];
      }
      elseif($item->file_host === 'local')
      {
        $zip        = new ZipArchive;
        $files_list = ['files' => []];
        $item_file  = get_main_file($item->file_name);

        if($zip->open($item_file) === TRUE)
        {
          for($i = 1; $i < $zip->numFiles; $i++ )
          { 
              $stat = $zip->statIndex($i); 

              $files_list['files'][] = ['name' => File::basename($stat['name']), 'mimeType' => File::extension($stat['name'])];
          }
        }
      }
      else
      {
        $files_list = [];
      }

      return response()->json($files_list);
    }



    // Newsletter
    public function subscribe_to_newsletter(Request $request)
    {
      $request->validate(['email' => 'required|email']);
      
      $subscription = Newsletter_Subscriber::insertOrIgnore(['email' => strip_tags($request->email)]);

      $request->session()->flash('newsletter_subscription_msg', ($subscription->id ?? null) 
                                                                ? __('Subscription done')
                                                                : __('You are already subscribed to our newsletter'));

      return redirect(($request->redirect ?? '/') . '#footer');
    }


    // Newsletter / unsubscribe
    public function unsubscribe_from_newsletter(Request $request)
    {
        if(Auth::check() && $request->query('email'))
        {
            $user = $request->user();

            if(md5($request->query('email')) && md5($user->email))
            {
                DB::delete("DELETE FROM newsletter_subscribers WHERE email = ?", [$user->email]);

                return redirect('/')->with(['user_message' => __('You have been unsubscribed from our newsletter. Thank you.')]);
            }
        }

        return redirect('/');
    }



    private function set_home_categories($limit = 20)
    {
        $categories    = config('categories.category_parents');
        $subcategories = config('categories.category_children');

        if($categories && $subcategories)
        {
            $_categories  = [];

            foreach($categories as $category)
            {
              if(!key_exists($category->id, array_keys($subcategories)))
                continue;

              foreach($subcategories[$category->id] as $subcategory)
              {
                $_categories[] = (object)['name' => $subcategory->name, 
                                          'url' => route('home.products.category', [$category->slug, $subcategory->slug])];
              }
            }

            shuffle($_categories);

            $_categories = array_slice($_categories, 0, $limit);

            Config::set('home_categories', $_categories);
        }
    }
    



    public function read_notifications(Request $request)
    {
        ctype_digit($request->notif_id) || abort(404);

        $user_id = Auth::id();

        $response = DB::update("UPDATE notifications SET users_ids = REPLACE(users_ids, CONCAT('|', ?, ':0|'), CONCAT('|', ?, ':1|')) 
                            WHERE users_ids REGEXP CONCAT('/|', ? ,':0|/') AND id = ?",
                            [$user_id, $user_id, $user_id, $request->notif_id]);

        return response()->json(['status' => $response]);
    }



    public function add_to_cart(Request $request)
    {
        $request->validate([
            'id'              => 'required|numeric', 
            'custom_price'    => 'nullable|numeric|gte:0', 
            'group_buy'       => 'nullable|numeric|in:0,1',
            'uuid'            => 'required|uuid',
            'extended_price'  => 'nullable|numeric|in:0,1',
            'buy_now'         => 'nullable|numeric|in:0,1',
        ]);

        $product = Product::select($this->product_columns)->where('id', $request->post('id'))->first() ?? abort(404);

        $cart_items = User_Shopping_Cart_Item::where('uuid', $request->input('uuid'))->where('item_id', $product->id)->count();
        $keys       = Key::where('product_id', $product->id)->where('user_id', null)->count();

        if($cart_items && $keys && ($cart_items == $keys))
        { 
            $cart_items = User_Shopping_Cart_Item::where('uuid', $request->post('uuid'))->get()->toArray();

            $cart_items = array_combine(array_column($cart_items, 'id'), $cart_items);

            return response()->json(['cart_items' => $cart_items, 'status' => false, 'message' => __('This product is out of stock')]);
        }

        $price_expiry = null;

        if((is_numeric($product->stock) && $product->stock == 0) || ($product->has_keys() > 0 && $product->keys(remaining:0)->count() == 0))
        {
            return response()->json(['status' => false, 'message' => __('This product is out of stock')]);
        }

        if($request->post('extended_price') === '0')
        {
            if($request->input('custom_price') && $product->minimum_price && ($request->input('custom_price') < $product->minimum_price))
            {
                return response()->json(['status' => false, 'message' => __('The custom price must be >= to :price', ['price' => $product->minimum_price])]);
            }
            
            if($product->minimum_price && $request->input('custom_price'))
            {
                $product->price = $request->input('custom_price');
            }
            elseif($product->price <= 0)
            {
                $product->price = 0;
            }
            elseif($product->has_promo())
            {
                $product->price = $product->promo_price;

                if($promo_time = json_decode($product->promo_time ?? '', true))
                {
                    $price_expiry = \Carbon\Carbon::parse($promo_time['to'])->timestamp;  
                }
                else
                {
                    $price_expiry = now()->addDays(365)->timestamp;
                }
            }
            elseif($request->post('groupBuy') === '1' && productHasGroupBuy($product))
            {
                $product->price        = $product->group_buy_price;
                $product->custom_price = $product->group_buy_price;

                $price_expiry = $product->group_buy_expiry;
            }

            $product->price = format_amount($product->price, false, config("payments.decimals.".currency('code'), 2));
        }
        else
        {
            $product->price = format_amount($product->extended_price, false, config("payments.decimals.".currency('code'), 2));
        }

        $product->cover = src("storage/covers/{$product->cover}");
        $product->url   = item_url($product);
        
        if($request->post('buy_now'))
        {
            User_Shopping_Cart_Item::where('uuid', $request->input('uuid'))->delete();
        }

        User_Shopping_Cart_Item::create([
            'item_id'         => $product->id,
            'category_id'     => $product->category->id,
            'category_name'   => $product->category->name,
            'category_url'    => category_url($product->category->slug),
            'summary'         => $product->short_description,
            'url'             => item_url($product),
            'extended_license' => $request->input('extended_price') === '1' ? 1 : 0,
            'name'            => $product->name,
            'price'           => $product->price,
            'custom_price'    => $request->input('custom_price'),
            'uuid'            => $request->input('uuid'),
            'cover'           => $product->cover,
            'price_expiry'    => $price_expiry,
        ]);

        if($request->post('buy_now'))
        {
            return response()->json(['status' => true, 'checkout_url' => route('home.checkout', ['type' => 'items', 'id' => $request->input('uuid')])]);  
        }

        $cart_items = User_Shopping_Cart_Item::where('uuid', $request->post('uuid'))->get()->toArray();

        $cart_items = array_combine(array_column($cart_items, 'id'), $cart_items);

        return response()->json(['cart_items' => $cart_items, 'status' => true]);
    }



    public function remove_from_cart(Request $request)
    {
        !Validator::make($request->post(), ['id' => 'required|numeric', 'uuid' => 'required|uuid'])->fails() ?? abort(404);

        User_Shopping_Cart_Item::where(['uuid' => $request->post('uuid'), 'id' => $request->post('id')])->delete();

        return ['status' => true];
    }


    public function load_cart_items(Request $request)
    {
        $cart_items = User_Shopping_Cart_Item::where('uuid', $request->post('uuid'))->get()->toArray();

        $cart_items = array_combine(array_column($cart_items, 'id'), $cart_items);

        foreach($cart_items as &$cart_item)
        {
            $cart_item['price'] = price($cart_item['price']);
        }

        return response()->json(['cart_items' => $cart_items]);
    }


    public function update_price(Request $request)
    {
      $request->validate(['items' => 'array|required']);

      $items = array_filter($request->post('items'));

      foreach($items as &$item)
      {
        $request->merge([
          'item' => [
            'id' => $item['id'],
            'license_id' => $item['license_id'] ?? null,
            'custom_price' => $item['custom_price'] ?? null
          ]
        ]);

        if(productHasGroupBuy((object)$item))
        {
          continue;
        }

        $item = $this->add_to_cart($request)->getData(true)['product'];
      }

      return response()->json(['items' => json_decode(json_encode($items), true)]);
    }


    public function get_group_buy_buyers(Request $request)
    {
      is_numeric($request->query('product_id')) ?? abort(404);

      $buyers_count = User_Shopping_Cart_Item::where(['user_ip' => $request->ip(), 'item_id' => $request->query('product_id')])->count();

      return json(['buyers' => $buyers_count]);
    }
                                 

    public static function init_notifications()
    {
      $notifications = [];

      if($user_id = Auth::id())
      {
        $notifications = DB::select("SELECT products.id as product_id, products.name, products.slug, notifications.updated_at, 
                                      notifications.id, `for`,
                                        CASE
                                          WHEN `for` = 0 THEN IFNULL(products.cover, 'default.webp')
                                          WHEN `for` = 1 OR `for` = 2 THEN IFNULL(users.avatar, 'default.webp')
                                        END AS `image`,
                                        CASE notifications.`for`
                                          WHEN 0 THEN 'New release is available for :product_name'
                                          WHEN 1 THEN 'Your comment has been approved for :product_name'
                                          WHEN 2 THEN 'Your review has been approved for :product_name'
                                        END AS `text`
                                       FROM notifications USE INDEX (users_ids, updated_at)
                                       JOIN products ON products.id = notifications.product_id
                                       JOIN users ON users.id = ?
                                       WHERE users_ids REGEXP CONCAT('/|', ? ,':0|/')
                                       ORDER BY updated_at DESC
                                       LIMIT 5", [$user_id, $user_id]);

        config(['notifications' => $notifications]);
      }

      return $notifications;
    }



    public function save_reaction(Request $request)
    {
      $this->middleware('auth');

      $request->validate([
        'product_id' => 'required|numeric',
        'item_id' => 'required|numeric',
        'item_type' => 'required|string',
        'reaction' => 'required|string|max:255'
      ]);

      $res = DB::insert("INSERT INTO reactions (product_id, item_id, item_type, user_id, name) VALUES (?, ?, ?, ?, ?) 
                  ON DUPLICATE KEY UPDATE name = ?", 
                  [$request->product_id, $request->item_id, $request->item_type, \Auth::id(), $request->reaction, $request->reaction]);

      $reactions = $this->get_reactions($request);

      return response()->json(['status' => $res, 'reactions' => $reactions]);
    }


    // Get item data for user favorites
    public function get_item_data(Request $request)
    {
      $request->validate(['id' => 'numeric|required|gt:0']);

      $item = Product::select($this->product_columns)->where('products.id', $request->query('id'))->first();

      $item->setAttribute('category_name', config("categories.category_parents.{$item->category_id}.name"));
      
      $item->setAttribute('category_slug', config("categories.category_parents.{$item->category_id}.slug"));
      
      $item->setAttribute('category_url', category_url($item->category_slug));

      $item->setAttribute('url', item_url($item));
      
      $item->setAttribute('cover', src("storage/covers/".($item->cover ?? "default.webp")));

      return json(compact('item')); 
    }


    // App installation
    public function install(Request $request)
    {      
      if(config('app.installed'))
      {
        return redirect('/');
      }

      if($request->method() === 'POST')
      {
        $request->validate([
          'database.*'          => 'required|bail|string',
          'site.name'           => 'required|bail|string',
          'site.title'          => 'required|bail|string',
          'site.items_per_page' => 'numeric|bail|string|gt:0',
          'site.purchase_code'  => 'required|bail|string',
          'admin.username'      => 'required',
          'admin.email'         => 'required|bail|email',
          'admin.password'      => 'required|bail|max:255',
          'admin.avatar'        => 'nullable|bail|image',
        ]);

        /** CREATE DATABASE CONNECTION START **/
          $db_params = $request->input('database');

          Config::set("database.connections.mysql", array_merge(config('database.connections.mysql'), $db_params));

          try 
          {
            DB::connection()->getPdo();
          }
          catch (\Exception $e)
          {
            $validator = Validator::make($request->all(), [])
                         ->errors()->add('Database', $e->getMessage());

            return redirect()->back()->withErrors($validator)->withInput();
          }
        /** CREATE DATABASE CONNECTION END **/


        /** CREATE DATABASE TABLES START **/
        DB::unprepared(File::get(base_path('database/database.sql')));
        /** CREATE DATABASE TABLES END **/



        /** SETTING .ENV VARS START **/
        update_env_var("DB_HOST", wrap_str($db_params['host']));
        update_env_var("APP_ENV", "production");
        update_env_var("DB_DATABASE", wrap_str($db_params['database']));
        update_env_var("DB_USERNAME", wrap_str($db_params['username']));
        update_env_var("DB_PASSWORD", wrap_str($db_params['password']));
        update_env_var("APP_NAME", wrap_str($request->input('site.name')));
        update_env_var("APP_URL", wrap_str("{$_SERVER['REQUEST_SCHEME']}://{$_SERVER['HTTP_HOST']}"));
        update_env_var("APP_INSTALLED", 'true');
        update_env_var("PURCHASE_CODE", wrap_str($request->input('site.purchase_code')));
        update_env_var("SESSION_DOMAIN", str_replace('.www', '', $_SERVER['HTTP_HOST']));
        update_env_var("SESSION_DRIVER", wrap_str('database'));
        update_env_var("CACHE_DRIVER", wrap_str('database'));
        /** SETTING .ENV VARS END **/


        /** CREATE ADMIN USER START **/
          if(!$user = User::where('email', $request->input('admin.email'))->first())
          {
            $user = new User;

            $user->name = $request->input('admin.username');
            $user->email = $request->input('admin.email');
            $user->password = Hash::make($request->input('admin.password'));
            $user->email_verified_at = date('Y-m-d');
            $user->role = 'admin';
            $user->avatar = 'default.webp';

            // Avatar
            if($avatar_file = $request->file('admin.avatar'))
            {
              $user_auto_inc_id = DB::select("SHOW TABLE STATUS LIKE 'users'")[0]->Auto_increment;

              $ext    = $avatar_file->extension();
              $avatar = $avatar_file->storeAs('avatars', "{$user_auto_inc_id}.{$ext}", ['disk' => 'public']);

              $user->avatar = pathinfo($avatar, PATHINFO_BASENAME);
            }

            $user->save();
          }
        /** CREATE ADMIN USER END **/


        $settings = Setting::first();

        /** GENERAL SETTINGS START **/
        //----------------------------
          $general_settings = json_decode($settings->general);

          $general_settings->name           = $request->input('site.name');
          $general_settings->title          = $request->input('site.title');
          $general_settings->description    = $request->input('site.description');
          $general_settings->items_per_page = $request->input('site.items_per_page');
          $general_settings->timezone       = $request->input('site.timezone');
          $general_settings->purchase_code  = $request->input('site.purchase_code');
          $general_settings->email          = $request->input('admin.email');
          
          $settings->general = json_encode($general_settings);
        /** GENERAL SETTINGS END **/


        /** MAILER SETTINGS START **/
        //----------------------------
          $mailer_settings = json_decode($settings->mailer);

          $mailer_settings->mail = json_encode($request->input('mailer.mail'));

          $mailer_settings = json_encode($mailer_settings);
        /** MAILER SETTINGS END **/


        $settings->save();

        Auth::loginUsingId($user->id, true);

        return redirect()->route('admin');
      }

      generate_app_key();

      //\Artisan::call('key:generate');

      $mysql_user_version = ['distrib' => __('Unable to find MySQL version'), 'version' => null, 'compatible' => false];

      if(function_exists('exec') || function_exists('shell_exec'))
      {
        $mysqldump_v = function_exists('exec') ? exec('mysqldump.exe --version') : shell_exec('mysqldump --version');
        
        preg_match('/(?<mysqlVersion>\d+\.\d+\.\d+(-\w+)?)/i', $mysqldump_v, $matches);

        if($mysqld = ($matches['mysqlVersion'] ?? null))
        {
          $mysql_user_version['distrib'] = (stripos($mysqld, 'mariadb') !== false) ? 'mariadb' : 'mysql';
          $mysql_user_version['version'] = (stripos($mysqld, 'mariadb') !== false) ? explode('-', $mysqld, 2)[0] : $mysqld;

          if($mysql_user_version['distrib'] == 'mysql' && $mysql_user_version['version'] >= 5.7)
          {
            $mysql_user_version['compatible'] = true;
          }
          elseif($mysql_user_version['distrib'] == 'mariadb' && $mysql_user_version['version'] >= 10)
          {
            $mysql_user_version['compatible'] = true;
          }
        }
      }

      $requirements = [
        "php" => ["version" => 8.0, "current" => phpversion()],
        "mysql" => ["version" => 8.0, "current" => $mysql_user_version],
        "php_extensions" => [
          "curl" => false,
          "fileinfo" => false,
          "intl" => false,
          "json" => false,
          "mbstring" => false,
          "openssl" => false,
          "mysqli" => false,
          "zip" => false,
          "ctype" => false,
          "dom" => false,
          "calendar" => false,
          "xml" => false,
          "xsl" => false,
          "pcre" => false,
          "tokenizer" => false
        ],
      ];

      $php_loaded_extensions = get_loaded_extensions();

      foreach($requirements['php_extensions'] as $name => &$enabled)
      {
          $enabled = in_array($name, $php_loaded_extensions);
      }

      return view('install', compact('requirements'));
    }



    public function get_reactions(Request $request)
    {
      if($request->users)
      {
        $reactions = Reaction::selectRaw("reactions.name, users.name as user_name, IFNULL(users.avatar, 'default.webp') as user_avatar")
                          ->join(DB::raw('users USE INDEX(primary)'), 'users.id', '=', 'reactions.user_id')
                          ->where(['reactions.item_id'    => $request->item_id, 
                                   'reactions.product_id' => $request->product_id,
                                   'reactions.item_type'  => $request->item_type]);

        //$reactions = $request->reaction ? $reactions->where('reactions.name', $request->reaction) : $reactions;

        $reactions = $reactions->get();

        return response()->json(['reactions' => $reactions->groupBy('name')->toArray()]);
      }
      else
      {
        return Reaction::selectRaw("COUNT(reactions.item_id) as `count`, reactions.name")
                          ->join(DB::raw('users USE INDEX(primary)'), 'users.id', '=', 'reactions.user_id')
                          ->where(['reactions.item_id'    => $request->item_id, 
                                   'reactions.product_id' => $request->product_id,
                                   'reactions.item_type'  => $request->item_type])
                          ->groupBy('reactions.name')
                          ->get()->pluck('count', 'name')->toArray();
      }
    }


    public function get_cities(Request $request)
    {
      config('app.products_by_country_city') || abort(404);

      $country = $request->post('country') ?? abort(404);
      $cities = config("app.countries_cities.{$country}") ?? abort(404);

      return response()->json(compact('cities'));
    }



    public function delete_comment(Request $request)
    {
      $comment = Comment::where(['id' => $request->id, 'user_id' => Auth::id()])->first() ?? abort(404);
  
      if(!$comment->parent)
      {
        $subcomments = Comment::where(['parent' => $comment->id])->get();
        
        if($subcomments->count())
        {
          $ids = array_column($subcomments->toArray(), 'id');

          Reaction::where('item_type', 'comment')->whereIn('item_id', $ids)->delete();
        
          foreach($subcomments as $subcomment)
          {
            $subcomment->delete();
          }
        }        
      }

      $comment->delete();

      return redirect($request->redirect ?? '/');
    }



    public function delete_review(Request $request)
    {
      $review = Review::where(['id' => $request->id, 'user_id' => Auth::id()])->first() ?? abort(404);

      $review->delete();

      return redirect($request->redirect ?? '/'); 
    }



    public function generate_sitemap(Request $request)
    {
      $sitemap = "";
      $type    = mb_strtolower(str_ireplace(['_', '.xml'], '', $request->type));

      $sitemap = sitemap($type);

      header('Content-Type: application/xml');

      exit($sitemap);
    }



    public function realtime_views(Request $request)
    {
      $user_id = md5($request->server("HTTP_USER_AGENT").'-'.$request->server("REMOTE_ADDR"));

      $realtime_views = [
        "website" => 0,
        "product" => 0
      ];

      if(config('app.realtime_views.website.enabled'))
      {
        if(!config('app.realtime_views.website.fake'))
        {
          $website_visitor_ids = \Cache::get('website_visitor_ids', []);

          $website_visitor_ids[$user_id] = time() + config('app.realtime_views.refresh', 5);

          foreach($website_visitor_ids as $visitor_id => $expire)
          {
            if(time() >= $expire)
            {
              unset($website_visitor_ids[$visitor_id]);   
            }
          }

          \Cache::forever('website_visitor_ids', $website_visitor_ids);

          $realtime_views['website'] = count($website_visitor_ids);
        }
        else
        {
          $fake_views_range = explode(',', config('app.realtime_views.website.range', '15,30'));

          $realtime_views['website'] = rand(...$fake_views_range);
        }
      }

      if(config('app.realtime_views.product.enabled') && is_numeric($request->query('i')))
      {
        if(!config('app.realtime_views.product.fake'))
        {
          $products_visitor_ids = \Cache::get('products_visitor_ids', []);
          $current_product_id   = $request->query('i');
          $product_visitor_ids  = $products_visitor_ids[$current_product_id] ?? [];

          $product_visitor_ids[$user_id] = time() + config('app.realtime_views.refresh', 5);

          foreach($product_visitor_ids as $visitor_id => $expire)
          {
            if(time() >= $expire)
            {
              unset($product_visitor_ids[$visitor_id]);   
            }
          }

          $products_visitor_ids[$current_product_id] = $product_visitor_ids;

          \Cache::forever('products_visitor_ids', $products_visitor_ids);

          $realtime_views['product'] = count($product_visitor_ids);
        }
        else
        {
          $fake_views_range = explode(',', config('app.realtime_views.product.range', '15,30'));

          $realtime_views['product'] = rand(...$fake_views_range);
        }
      }

      header("Cache-Control: no-cache, no-store, must-revalidate");
      header("Content-Type: application/javascript");

      if($realtime_views['product'] > $realtime_views['website'])
      {
        $realtime_views['website'] = $realtime_views['product'] + rand(10, 20);
      }

      $realtime_views = json_encode($realtime_views, JSON_PRETTY_PRINT);

      exit(<<<SCRIPT
      app.realtimeViews = $realtime_views;
      SCRIPT); 
    }



    public function bricks_mask(Request $request)
    {
        return cover_mask($request->query('name'));
    }


    public function stream_video(Request $request)
    {
      $product = Product::find($request->id) ?? abort(404);

      $supported_formats = [
        //"webv" => "video/webm",
        "mp4" => "video/mp4"
      ];

      if(isset($supported_formats[$product->file_extension]))
      {
        if($product->file_host === "local")
        {
          $file = storage_path("app/downloads/{$product->file_name}");

          if(file_exists($file))
          {
            $vStream = new \App\Libraries\VideoStream($file);

            exit($vStream->start());
          }
        }
        elseif($product->file_host === "yandex")
        {
          try
          {
            $vStream = new \App\Libraries\VideoStreamUrl(urldecode(base64_decode($request->temp_url)));

            exit($vStream->start());
          }
          catch(\Exception $e)
          {

          }
        }
      }
    }


    public function set_template(Request $request)
    {
      $url  = urldecode($request->query('redirect', ''));

      if(auth_is_admin() || env_is('local'))
      {
        $template = $request->query('template');

        $templates = \File::glob(resource_path('views/front/*', GLOB_ONLYDIR));
        $base_path = resource_path('views/front/');
        $templates = array_filter($templates, 'is_dir');
        $templates = str_ireplace($base_path, '', $templates);

        if(in_array($template, $templates))
        {
            session(["template" => $template]);
        }
      }

      return redirect($url);
    }


    public function set_currency(Request $request)
    {
      if(config('app.installed') === true)
      {
        $url  = urldecode($request->query('redirect', ''));
        $code = $request->query('code');

        if(in_array(mb_strtoupper($code), array_keys(config('payments.currencies', []))))
        {
            session(["currency" => $code]);
        }
      }
      
      return redirect($url);
    }


    public function set_locale(Request $request)
    {
        $url  = $request->post('redirect', '');
        $locale = $request->post('locale', config('app.locale'));

        if(in_array($locale, \LaravelLocalization::getSupportedLanguagesKeys()))
        {
            session(["locale" => $locale]);
        }

        return redirect($url);
    }


    public function update_statistics(Request $request)
    {
      try
      {
          if(!$stats = \App\Models\Statistic::where("date", \DB::raw("CURDATE()"))->first())
          {
              $stats = new \App\Models\Statistic;
          }

          // Traffic
          if($code = user_country($request->ip(), "isoCode"))
          {
              $stats->traffic = implode(",", array_filter(array_merge(explode(',', $stats->traffic), [$code])));
          }

          if(!BrowserDetect::isBot())
          {
              // Browser name
              if($browser = BrowserDetect::browserFamily())
              {
                  $browsers = json_decode($stats->browsers, true) ?? [];

                  if(isset($browsers[$browser]))
                  {
                      $browsers[$browser] += 1;  
                  }
                  else
                  {
                      $browsers[$browser] = 1;
                  }

                  $stats->browsers = json_encode($browsers);
              }

              // Operating systems
              if($os = BrowserDetect::platformFamily())
              {
                  $oss = json_decode($stats->oss, true) ?? [];

                  if(isset($oss[$os]))
                  {
                      $oss[$os] += 1;  
                  }
                  else
                  {
                      $oss[$os] = 1;
                  }

                  $stats->oss = json_encode($oss);
              }

              // Devices (mobile, tablet, desktop)
              $devices = json_decode($stats->devices, true) ?? [];
              $devices = array_merge(['mobile' => 0, 'tablet' => 0, 'desktop' => 0], $devices);

              if(BrowserDetect::isMobile())
              {
                  $devices['mobile'] += 1;  
              }
              elseif(BrowserDetect::isTablet())
              {
                  $devices['tablet'] += 1;
              }
              elseif(BrowserDetect::isDesktop())
              {
                  $devices['desktop'] += 1;
              }

              $stats->devices = json_encode($devices);
          }

          $stats->date = \DB::raw("CURDATE()");

          $stats->save();
      }
      catch(\Throwable $e)
      {
        
      }
    }


    public function load_translations()
    {
        $seconds_to_cache = 3600;
        $ts = gmdate("D, d M Y H:i:s", time() + $seconds_to_cache) . " GMT";
        header("Expires: $ts");
        header("Pragma: cache");
        header("Cache-Control: max-age=$seconds_to_cache");
        header("Content-Type: text/javascript");

        exit("window.translation = ".json_encode(config('translation')));
    }


    public function load_js_props(Request $request)
    {
        $js_code = (string)view('components.js_props', ['payment_processor' => $request->query('processor')]);

        $seconds_to_cache = 3600;
        $ts = gmdate("D, d M Y H:i:s", time() + $seconds_to_cache) . " GMT";
        header("Expires: $ts");
        header("Pragma: cache");
        header("Cache-Control: max-age=$seconds_to_cache");
        header("Content-Type: application/javascript");

        exit($js_code);
    }




    public function user_coupons(Request $request)
    { 
        if(!intval(config('app.user_coupons_page')))
        {
            abort(404);
        }

        $coupons =  Coupon::selectRaw('*, IF(coupons.expires_at IS NOT NULL AND expires_at < NOW(), 1, 0) as expired')->where(function($query)
                    {
                      $query->where("users_ids", "=", null)
                            ->orWhere("users_ids", "=", "")
                            ->orWhereRaw("users_ids REGEXP QUOTE(?)", [Auth::id()]);
                    })
                    ->orderBy('id', 'desc')
                    ->paginate(10);

        $user = $request->user();

        return view_("user.coupons", compact('coupons', 'user'));
    }



    // User Purchases
    public function user_purchases(Request $request)
    {
        $transactions = Transaction::where(['user_id' => Auth::id(), 'type' => 'product']);

        if(!config('app.allow_download_in_test_mode'))
        {
            $transactions = $transactions->where('sandbox', 0);
        }
        
        $transactions = $transactions->orderBy('id', 'desc')->paginate(10);

        $transactions_collection = $transactions->getCollection();

        foreach($transactions_collection as &$transaction)
        {
            $items = json_decode($transaction->details, true);
            $items = $items['items'] ?? [];

            $checkout_controller = new \App\Http\Controllers\CheckoutController();

            $transaction->setAttribute('items', $checkout_controller->order_download_links($transaction, 0));

            if($items)
            {
                foreach($transaction->items ?? [] as $item)
                {
                    foreach($items as $_item)
                    {
                        if($item->id === $_item['item_id'])
                        {
                            $item->cover = $_item['cover'];
                        }
                    }
                }
            }
        }

        $transactions->setCollection($transactions_collection);

        $user = $request->user();

        config(['meta_data.title' => __('Purchases')]);

        return view_("user.purchases", compact('transactions', 'user'));
    }


    public function user_collection(Request $request)
    {
        $user = $request->user();

        config(['meta_data.title' => __('Collection')]);

        return view_("user.collection", compact('user'));
    }


    public function load_user_collection(Request $request)
    {
        if(!$favorites = $request->post('favorites'))
        {
            return response()->json(['status' => false]);
        }

        $ids = array_column($favorites, 'id');

        $items = [];

        if(count($ids))
        {
            $items = Product::select($this->product_columns)->whereIn('products.id', $ids)->get()->map(function($item)
            {
                $item->setAttribute('category_name', config("categories.category_parents.{$item->category_id}.name"));

                $item->setAttribute('category_slug', config("categories.category_parents.{$item->category_id}.slug"));
                
                $item->setAttribute('category_url', category_url($item->category_slug));

                $item->setAttribute('url', item_url($item));
                
                $item->setAttribute('cover', src("storage/covers/".($item->cover ?? "default.webp")));

                return $item->only('id', 'name', 'category_name', 'url', 'category_url', 'cover');
            })->toArray();
        }

        return response()->json(['status' => true, 'items' => (object)$items]);
    }


    public function user_dashboard(Request $request)
    { 
        config(['meta_data.title' => __('Dashboard')]);

        $credits = user_credits(true);

        $user            = $request->user();
        $orders          = Transaction::where('user_id', Auth::id())->count();
        $comments        = Comment::where('user_id', Auth::id())->count();
        $reviews         = Review::where('user_id', Auth::id())->count();
        $subscriptions   = User_Subscription::where('user_id', Auth::id())->count();
        $credits         = $credits['total_available_credits'] ?? 0;
        $referred_users  = DB::select("SELECT COUNT(DISTINCT referee_id) as _count FROM affiliate_earnings WHERE referrer_id = ?", [Auth::id()])[0]->_count ?? 0 ;
        $last_orders     = Transaction::where('user_id', Auth::id())->orderBy('id', 'desc')->limit(5)->get();

        return view_('user.dashboard', compact('user', 'orders', 'comments', 'reviews', 'subscriptions', 'credits', 'referred_users', 'last_orders'));
    }


    public function user_prepaid_credits(Request $request)
    {
        config(['meta_data.title' => __('Prepaid credits')]);

        $expires_in_days = config('app.prepaid_credits.expires_in', null);

        $user_prepaid_credits = User_Prepaid_Credit::useIndex('user_id')->with('prepaid_credit')
                                ->selectRaw('prepaid_credits.name, prepaid_credits.discount, prepaid_credits.amount, user_prepaid_credits.prepaid_credits_id, user_prepaid_credits.id, user_prepaid_credits.credits, 
                                transactions.status, IF(? IS NOT NULL, NOW() >= DATE_ADD(user_prepaid_credits.updated_at, INTERVAL ? DAY), 0) as expired,
                                IF(? IS NOT NULL, DATE_ADD(user_prepaid_credits.updated_at, INTERVAL ? DAY), null) as expires_at,
                                user_prepaid_credits.updated_at', [$expires_in_days, $expires_in_days, $expires_in_days, $expires_in_days])
                                ->join('transactions', 'transactions.id', '=', 'user_prepaid_credits.transaction_id')
                                ->join('prepaid_credits', 'user_prepaid_credits.prepaid_credits_id', '=', 'prepaid_credits.id')
                                ->where('user_prepaid_credits.user_id', auth()->user()->id)
                                ->orderBy('user_prepaid_credits.id', 'DESC')
                                ->paginate(15);

        foreach($user_prepaid_credits->items() as &$item)
        {
            if($item->status === 'paid')
            {
              if($item->expired)
              {
                  $item->status = 'expired';
              }
              else
              {
                  $item->status = 'active';
              }
            }
            else
            {
                $item->status = 'pending';
            }
        }

        $user = $request->user();

        return view_('user.prepaid_credits', compact('user_prepaid_credits', 'user'));
    }


    public function user_security(Request $request)
    {
        config(['meta_data.title' => __('Security')]);

        $user = User::find($request->user()->id);

        if($request->method() === 'POST')
        {
            $request->validate([
                'old_password'    => 'string|nullable|max:255|bail',
                'new_password'    => 'string|nullable|max:255|bail',
                'two_factor_auth' => 'nullable|numeric|in:0,1',
            ]);

            $user->two_factor_auth = $request->post('two_factor_auth') ?? '0';

            if($request->post('old_password') && $request->post('new_password'))
            {
                Validator::make($request->post(), [
                  'old_password' => [
                      function ($attribute, $value, $fail) 
                      {
                          if(! Hash::check($value, auth()->user()->password)) {
                              $fail($attribute.' '. __('is incorrect'));
                          }
                      }
                  ],
                ])->validate();
              
                $user->password = Hash::make($request->new_password);
            }

            $user->save();

            return redirect()->route('user.security');
        }

        return view_('user.security', compact('user'));
    }


    // Profile
    public function user_profile(Request $request)
    {
        config(['meta_data.title' => __('Profile')]);

        $user = User::find($request->user()->id);

        if($request->method() === 'POST')
        {
            $cashout_methods = implode(',', array_values(config('affiliate.cashout_methods', [])));

            $request->validate([
                'name' => 'string|nullable|max:255|bail',
                'firstname' => 'string|nullable|max:255|bail',
                'lastname' => 'string|nullable|max:255|bail',
                'country' => 'string|nullable|max:255|bail',
                'city' => 'string|nullable|max:255|bail',
                'address' => 'string|nullable|max:255|bail',
                'zip_code' => 'string|nullable|max:255|bail',
                'id_number' => 'string|nullable|max:255|bail',
                'state' => 'string|nullable|max:255|bail',
                'affiliate_name' => 'string|nullable|max:255|bail',
                'cashout_method' => "string|nullable|in:{$cashout_methods}|max:255|bail",
                'paypal_account' => 'string|nullable|email|max:255|bail',
                'bank_account' => 'array|nullable|bail',
                'bank_account.*' => 'nullable|string|bail',
                'phone' => 'string|nullable|max:255|bail',
                'receive_notifs' => 'string|nullable|in:0,1|bail',
                'old_password' => 'string|nullable|max:255|bail',
                'new_password' => 'string|nullable|max:255|bail',
                'avatar' => 'nullable|image',
                'credits_sources' => 'nullable|string',
                'two_factor_auth' => 'nullable|numeric|in:0,1',
            ]);

            if($affiliate_name = $request->post('affiliate_name'))
            {
                if(User::where('affiliate_name', $affiliate_name)->where('id', '!=', $user->id)->first())
                {
                    return back_with_errors(['user_message' => __('The affiliate name is already token, please chose another one.')]);
                }
            }

            $user->name             = $request->input('name', $user->name ?? null);
            $user->firstname        = $request->input('firstname', $user->firstname ?? null);
            $user->lastname         = $request->input('lastname', $user->lastname ?? null);
            $user->country          = $request->input('country', $user->country ?? null);
            $user->city             = $request->input('city', $user->city ?? null);
            $user->address          = $request->input('address', $user->address ?? null);
            $user->zip_code         = $request->input('zip_code', $user->zip_code ?? null);
            $user->id_number        = $request->input('id_number', $user->id_number ?? null);
            $user->state            = $request->input('state', $user->state ?? null);
            $user->affiliate_name   = $request->input('affiliate_name');
            $user->paypal_account   = $request->input('paypal_account');
            $user->bank_account     = json_encode($request->input('bank_account'));
            $user->phone            = $request->input('phone', $user->phone ?? null);
            $user->receive_notifs   = $request->input('receive_notifs', $user->receive_notifs ?? '1');
            $user->cashout_method   = $request->input('cashout_method');
            $user->credits_sources  = $request->input('credits_sources');
            $user->two_factor_auth  = $request->post('two_factor_auth') ?? '0';

            if($request->old_password && $request->new_password)
            {
                Validator::make($request->all(), [
                  'old_password' => [
                      function ($attribute, $value, $fail) 
                      {
                          if(! Hash::check($value, auth()->user()->password)) {
                              $fail($attribute.' is incorrect.');
                          }
                      }
                  ],
                ])->validate();
              
                $user->password = Hash::make($request->new_password);
            }


            if($avatar = $request->file('avatar'))
            {
                $request->validate(['avatar' => 'image']);

                $ext = mb_strtolower($avatar->extension());

                if($ext !== "svg")
                {
                    $ext = "webp";

                    if(extension_loaded('imagick'))
                    {
                        $img = ImageManager::imagick();
                    }
                    else
                    {
                        $img = ImageManager::gd();
                    }
                    
                    $img = $img->read($avatar);

                    $img->toWebp(100)->save("storage/avatars/{$user->id}.{$ext}");

                    $user->avatar = "{$user->id}.{$ext}";
                }
                else
                {
                    $avatar->storeAs('avatars', "{$user->id}.{$ext}", ['disk' => 'public']);

                    $user->avatar = "{$user->id}.{$ext}";
                }
            }

            $user->save();

            $request->session()->flash('profile_updated', __('Done').'!');

            return redirect()->route('user.profile');
        }

        $user = (object)$user->getAttributes();

        $user->fullname = null;

        if($user->firstname && $user->lastname)
        {
            $user->fullname = $user->firstname . ' ' . $user->lastname;
        }

        $user->bank_account = json_decode($user->bank_account);

        $credits_sources = [];

        if(config('affiliate.enabled'))
        {
            $credits_sources['affiliate_credits'] = __('Affiliate earnings');
        }

        if(config('app.prepaid_credits.enabled'))
        {
            $credits_sources['prepaid_credits'] = __('Prepaid credits');
        }
        
        return view_('user.profile', compact('user', 'credits_sources'));
    }


    // User invoices
    public function user_invoices(Request $request)
    {
        config(['meta_data.title' => __('Invoices')]);

        $invoices = Transaction::useIndex('user_id', 'confirmed')
                    ->select('id', 'reference_id', 'amount', 'created_at', 'details')
                    ->where(['user_id' => Auth::id(), 'confirmed' => 1])
                    ->where('details', '!=', null)
                    ->orderBy('id', 'desc')
                    ->paginate(10);

        foreach($invoices as &$invoice)
        {
            $details = json_decode($invoice->details);
            $currency = $details->currency ?? currency('code');

            $invoice->setAttribute('currency', $currency);
            $invoice->amount = $details->total_amount ?? $invoice->amount;
        }

        $user = $request->user();

        return view_("user.invoices", compact('invoices', 'user'));
    }


    // Favorites
    public function user_favorites(Request $request)
    {
        config(["meta_data.title" => __('Collection')]);

        return view_('user.collection');
    }


    // User subscriptions
    public function user_subscriptions(Request $request)
    {
        config(['meta_data.title' => __('Subscriptions')]);

        $user_subscriptions = User_Subscription::useIndex('user_id')
                              ->selectRaw("pricing_table.name, user_subscription.id, user_subscription.downloads, pricing_table.limit_downloads, user_subscription.starts_at, user_subscription.ends_at,
                                user_subscription.daily_downloads, pricing_table.limit_downloads_per_day,
                                IF(DATEDIFF(user_subscription.ends_at, CURRENT_TIMESTAMP) > 0, DATEDIFF(user_subscription.ends_at, CURRENT_TIMESTAMP), 0) as remaining_days,
                                ((user_subscription.ends_at IS NOT NULL AND CURRENT_TIMESTAMP > user_subscription.ends_at) OR
                                (pricing_table.limit_downloads > 0 AND user_subscription.downloads >= pricing_table.limit_downloads) OR 
                                (pricing_table.limit_downloads_per_day > 0 AND user_subscription.daily_downloads >= pricing_table.limit_downloads_per_day AND user_subscription.daily_downloads_date = CURDATE())) AS expired, transactions.status = 'paid' as payment_status")
                              ->join('pricing_table', 'pricing_table.id', '=', 'user_subscription.subscription_id')
                              ->join(DB::raw('transactions USE INDEX(primary)'), 'user_subscription.transaction_id', '=', 'transactions.id')
                              ->where('user_subscription.user_id', auth()->user()->id)
                              ->orderBy('user_subscription.starts_at', 'DESC')
                              ->paginate(10);

        $user = $request->user();

        return view_('user.subscriptions', compact('user_subscriptions', 'user'));
    }


    public function user_notifications(Request $request)
    {
        config(['meta_data.title' => __('Notifications')]);

        $notifications = Notification::useIndex('users_ids', 'updated_at')
                              ->selectRaw("products.id as product_id, products.name, products.slug, notifications.updated_at, notifications.id, `for`,
                                            CASE
                                              WHEN `for` = 0 THEN IFNULL(products.cover, 'default.png')
                                              WHEN `for` = 1 OR `for` = 2 THEN IFNULL(users.avatar, 'default.webp')
                                            END AS `image`,
                                            CASE notifications.`for`
                                              WHEN 0 THEN 'New version is available for :product_name'
                                              WHEN 1 THEN 'Your comment has been approved for :product_name'
                                              WHEN 2 THEN 'Your review has been approved for :product_name'
                                            END AS `text`,
                                            IF(users_ids LIKE CONCAT('%|', ?,':0|%'), 0, 1) AS `read`", [Auth::id()])
                              ->leftJoin('products', 'products.id', '=', 'notifications.product_id')
                              ->leftJoin('users', 'users.id', '=', DB::raw(Auth::id()))
                              ->where('users_ids', 'REGEXP', '\|'.Auth::id().':(0|1)\|')
                              ->where('products.slug', '!=', null)
                              ->orderBy('updated_at', 'desc')
                              ->paginate(5);

        $user = $request->user();

        return view_('user.notifications', compact('notifications', 'user'));
    }


    public function two_factor_authentication(Request $request)
    {
      $user = $request->user();

      if(!config('app.two_factor_authentication') || !$user->two_factor_auth || !$request->query('2fa_sec'))
      {
          abort(404);
      }

      if($request->isMethod('POST'))
      {
        if($request->post('2fa_sec') !== session('2fa_sec'))
        {
          return redirect('/');
        } 

        $request->validate(['verification_code.*' => 'required|numeric']);

        $code = array_filter($request->post('verification_code'), fn($input) => is_numeric($input));

        if(count($code) !== 6)
        {
          return back()->with(['user_message' => __('The verification code must contain 6 digits')]);
        }

        $code = implode('', $code);

        if(!verifyQRCode($request->user()->two_factor_auth_secret, $code))
        {
          return back()->with(['user_message' => __("Wrong code, please try again")]);
        }

        $user->update(['two_factor_auth_expiry' => config('app.two_factor_authentication_expiry') > 0 ? time()+(int)(config('app.two_factor_authentication_expiry')*60) : null]);
        
        return redirect($request->query('redirect', '/'));
      }

      config([
        "meta_data.name"  => __('Two Factor Authentication'),
        "meta_data.title" => __('Two Factor Authentication'),
      ]);

      $qrCodeUrl = null;

      if(!$request->user()->two_factor_auth_secret)
      {
        $response = generateQRCode(16, 200);

        $request->user()->update([
          'two_factor_auth_expiry' => null,
          'two_factor_auth_secret' => $response['secretKey'],
          'two_factor_auth_ip'     => $request->ip(),
        ]);

        $qrCodeUrl = $response['qrCodeUrl'];
      }

      return view('auth.two_factor_authenication', ['qrCodeUrl' => $qrCodeUrl]);
    }
}
