<?php

namespace App\Libraries;

use net\authorize\api\contract\v1 as AnetAPI;
use net\authorize\api\controller as AnetController;
use net\authorize\api\constants\ANetEnvironment;
use Illuminate\Support\Facades\{ DB, Session, Cache, Validator, Auth };
use Illuminate\Http\Request;
use App\Models\{ Transaction };
use GuzzleHttp\{ Client };

class Authorizenet 
{
    public $name = 'authorizenet';
    public $return_url;
    public $cancel_url;
    public $supported_currencies = ["USD", "CAD", "CHF", "DKK", "EUR", "GBP", "NOK", "PLN", "SEK", "AUD", "NZD"];
    public $currency_code;
    public $exchange_rate = 1;
    public $decimals;
    public $details  = [];
    public $error_msg;
    public static $response = "default";


		public function __construct()
		{
  			if(!config("payments_gateways.{$this->name}.enabled"))
  			{
  				  return response()->json(['user_message' => __(':payment_proc is not enabled', ['payment_proc' =>  'Authorizenet'])]);
  			}
              
  			$this->currency_code = config('payments.currency_code');
  			$this->decimals = config("payments.currencies.{$this->currency_code}.decimals");

  			prepare_currency($this);

        $this->details = [
          	"items" => [],
    	      "total_amount" => 0,
    	      "currency" => $this->currency_code,
    	      "exchange_rate" => $this->exchange_rate,
    	      "custom_amount" => null,
            "reference" => generate_transaction_ref(),
            "transaction_id" => null,
            "order_id" => null,
  	    ];

        $this->return_url = route("home.checkout.order_completed", ['processor' => $this->name, 'ref' => $this->details['reference']]);
        $this->cancel_url = route('home');
		}



		public function create_order(float $amount, array $payload)
		{
        $amount = format_amount($amount * $this->exchange_rate, false, $this->decimals);
        
        $this->details['total_amount'] = $amount;
        
        $merchantAuthentication = new AnetAPI\MerchantAuthenticationType();
        $merchantAuthentication->setName(config("payments_gateways.{$this->name}.api_login_id"));
        $merchantAuthentication->setTransactionKey(config("payments_gateways.{$this->name}.transaction_key"));

        $api_url = config("payments_gateways.{$this->name}.mode") == 'sandbox' ? ANetEnvironment::SANDBOX : ANetEnvironment::PRODUCTION;

        $OpaqueDataType = new AnetAPI\OpaqueDataType();
        $OpaqueDataType->setDataDescriptor($payload['opaqueData']['dataDescriptor']);
        $OpaqueDataType->setDataValue($payload['opaqueData']['dataValue']);

        $paymentOne = new AnetAPI\PaymentType();

        $paymentOne->setOpaqueData($OpaqueDataType);
        

        $transactionRequestType = new AnetAPI\TransactionRequestType();
        $transactionRequestType->setTransactionType("authCaptureTransaction"); 
        
        $lineItem = new AnetAPI\LineItemType();
        $lineItem->setItemId(time());
        $lineItem->setName(__('Purchase_from_:app_name', ['app_name' => mb_ucfirst(config('app.name'))]));
        $lineItem->setQuantity(1);
        $lineItem->setUnitPrice($amount);
        $lineItem->setTotalAmount($amount);

        $transactionRequestType->addToLineItems($lineItem);

        $transactionRequestType->setAmount($amount);
        $transactionRequestType->setCurrencyCode($this->currency_code);
        $transactionRequestType->setPayment($paymentOne);

        $request = new AnetAPI\CreateTransactionRequest();
        $request->setMerchantAuthentication($merchantAuthentication);
        $request->setRefId(generate_transaction_ref());
        
        $request->setTransactionRequest($transactionRequestType);

        $controller = new AnetController\CreateTransactionController($request);

        $apiResponse = $controller->executeWithApiResponse($api_url);

        if(strtolower($apiResponse->getMessages()->getResultCode() ?? '') === "error")
        {
            $this->error_msg = ['user_message' => $apiResponse->getMessages()->getMessage()[0]->getText()];
        }

        return $apiResponse->getTransactionResponse();
		}



    public function init_payment(float $amount)
    { 
        $validator = Validator::make(request()->all(), [
            'messages.resultCode'           => 'required|string|in:Ok',
            'encryptedCardData.cardNumber'  => 'required|string',
            'encryptedCardData.expDate'     => 'required|string',
            'encryptedCardData.bin'         => 'required|string',
            'customerInformation.firstName' => 'required|string',
            'customerInformation.lastName'  => 'required|string',
            'opaqueData.dataDescriptor'     => 'required|string',
            'opaqueData.dataValue'          => 'required|string',
        ]);

        if($validator->fails())
        {
            return ['user_message' => implode(',', $validator->errors()->all())];
        }

        $payload = request()->post();

        $authorize_net = new Authorizenet;

        if($authorize_net->error_msg)
        {
            return $authorize_net->error_msg;
        }

        $response = $authorize_net->create_order($amount, $payload);

        if($authorize_net->error_msg)
        {
            return $authorize_net->error_msg;
        }

        if($response->getResponseCode() != 1)
        {
            $errors = $response->getErrors();

            foreach($errors as &$error)
            {
                $error = $error->getErrorText();
            }

            return ['user_message' => json_encode($errors)];
        }

        return $this->return_url;
    }



    public function complete_payment(Request $request)
    {
        if(stripos($request->processor, $this->name) !== false && $request->get('ref') !== null)
        {
            $transaction_id = $request->get('ref');

            if(is_null($transaction_id))
            {
                return [
                    'status' => false, 
                    'user_message' => __('Missing transaction order_id.')
                ];
            }

            $transaction =  Transaction::where(['processor' => $this->name])
                            ->where(function($builder) use($transaction_id)
                            {
                                $builder->where(['transaction_id' => $transaction_id])
                                        ->orWhere(['order_id' => $transaction_id])
                                        ->orWhere(['reference_id' => $transaction_id]);
                            })
                            ->first();

            if(!$transaction)
            {
                return [
                    'status' => false, 
                    'user_message' => __('Missing transaction database record [:transaction_id].', ['transaction_id' => $transaction_id])
                ];
            }

            $transaction->status = 'pending';
            $transaction->save();
          
            return ['status' => true, 'user_message' => null, 'transaction' => $transaction];
        }

        return ['status' => false, 'user_message' => __('Something wrong happened.')];
    }
}





















