<?php

	namespace App\Libraries;

  use Illuminate\Support\Facades\{ DB, Session, Cache, Validator, Auth };
  use Illuminate\Http\Request;
  use App\Models\{ Transaction };
  use GuzzleHttp\{ Client };

	class Mollie 
	{		
		public $name = 'mollie';
    public $return_url;
    public $cancel_url;
    public $supported_currencies = ["EUR", "GBP", "DKK", "NOK", "PLN", "SEK", "CHF", "USD"];
    public $supported_locales = ["en_US", "en_GB", "nl_NL", "nl_BE", "fr_FR", "fr_BE", "de_DE", "de_AT", "de_CH", "es_ES", "ca_ES", "pt_PT", "it_IT", "nb_NO", "sv_SE", "fi_FI", "da_DK", "is_IS", "hu_HU", "pl_PL", "lv_LV", "lt_LT"];
    public $payment_methods = [
			"applepay",
			"bancontact",
			"banktransfer",
			"belfius",
			"creditcard",
			"directdebit",
			"eps",
			"giftcard",
			"giropay",
			"ideal",
			"kbc",
			"mybank",
			"paypal",
			"paysafecard",
			"przelewy24",
			"sofort",
    ];
    public $currency_code;
    public $exchange_rate = 1;
    public $decimals;
    public $details  = [];
    public $error_msg;
    public $api_key;
    public $locale = 'en_US';
    public static $response = "default";


    public function __construct(string $locale = null)
    {
      exists_or_abort(config("payments_gateways.{$this->name}"), __(':payment_proc is not enabled', ['payment_proc' =>  $this->name]));

      $this->api_key  = config("payments_gateways.{$this->name}.api_key");
      $this->currency_code = config('payments.currency_code');
      $this->decimals = config("payments.currencies.{$this->currency_code}.decimals", 2);

      if($payment_methods = array_filter(explode(',', config("payments_gateways.{$this->name}.method"))))
      {
      	$this->payment_methods = $payment_methods;
      }

      prepare_currency($this);

      if($locale && in_array($locale, $this->supported_locales))
      {
        $this->locale = $locale;
      }

      $this->details = [
        'items' => [],
        'total_amount' => 0,
        'currency' => $this->currency_code,
        'exchange_rate' => $this->exchange_rate,
        'custom_amount' => null,
        'reference' => generate_transaction_ref(),
        'transaction_id' => null,
        'order_id' => null,
      ];

      $this->return_url  = route('home.checkout.order_completed', ['processor' => $this->name, 'ref' => $this->details['reference']]);
      $this->cancel_url  = route('home');
      $this->webhook_url = route('home.checkout.webhook', ['processor' => $this->name, 'ref' => $this->details['reference']]);
    }



		public function create_payment_link(float $amount)
		{
  			$amount = format_amount($amount * $this->exchange_rate, false, $this->decimals);

        $this->details['total_amount'] = $amount;

        $payload = [
          	"amount" => [
            		"currency" => $this->currency_code,
            		"value" => $amount,
          	],
          	"description" =>  __("Purchase from :app_name", ["app_name" => mb_ucfirst(config("app.name"))]),
          	"redirectUrl" => $this->return_url,
          	"cancelUrl" => $this->cancel_url,
          	"webhookUrl" => $this->webhook_url,
          	"locale" => $this->locale,
          	"method" => $this->payment_methods,
        ];

        $payload = array_filter($payload);

        $client = new Client([
          	"verify" => false,
          	"http_errors" => false,
          	"headers" => [
          		  "Authorization" => "Bearer {$this->api_key}"
          	]
        ]);

        $response = $client->request("POST", "https://api.mollie.com/v2/payments", ["json" => $payload]);

        if(stripos($response->getStatusCode(), '20') === false)
        {
        		$this->error_msg = ['user_message' => (string)$response->getBody()];

          	return;
        }

        $response = json_decode((string)$response->getBody(), true);

        return $response;
		}



    public function init_payment(float $amount)
    {
      $response = $this->create_payment_link($amount);

      if($this->error_msg)
      {
          return $this->error_msg;
      }

      $url = $response['_links']['checkout']['href'] ?? abort(404);

      return $url;
    }



    public function complete_payment(Request $request)
    {
      if(stripos($request->processor, $this->name) !== false && $request->ref !== null)
      {
          $transaction_id = $request->ref;

          $transaction =  Transaction::where(['processor' => $this->name])
                          ->where(function($builder) use($transaction_id)
                          {
                            $builder->where(['transaction_id' => $transaction_id])
                                    ->orWhere(['order_id' => $transaction_id])
                                    ->orWhere(['reference_id' => $transaction_id]);
                          })
                          ->first();

          if(!$transaction)
          {
              return [
                  'status' => false, 
                  'user_message' => __('Missing transaction database record [:transaction_id].', ['transaction_id' => $transaction_id])
              ];
          }

          if($transaction->status !== 'paid')
          {
              $transaction->status = 'pending';

              $transaction->save();
          }

          return ['status' => true, 'user_message' => null, 'transaction' => $transaction];
      }

      return ['status' => false, 'user_message' => __('Something wrong happened.')];
    }

    
    public function verify_payment($transaction)
    {      
        $api_url = "https://api.mollie.com/v2/payments/{$transaction->transaction_id}";

        $client = new Client([
          	"verify" => false,
          	"http_errors" => false,
          	"headers" => [
          		  "Authorization" => "Bearer {$this->api_key}"
          	]
        ]);

        $response = $client->request("GET", $api_url);

        if(stripos($response->getStatusCode(), '20') === false)
        {
        		$this->error_msg = ['user_message' => (string)$response->getBody()];

          	return;
        }

        $response = json_decode((string)$response->getBody(), true);

        $status = strtolower($response['status'] ?? '');

        if($status === "paid")
        {
        	 return ['status' => true, 'response' => $response];
        }

       	return ['status' => false];
    }


    public function handle_webhook_notif(Request $request)
    {
        $response = ['status' => 0, 'transaction' => null, 'valid' => 0];

        if(stripos($request->processor, $this->name) !== false && $request->post('id') && $request->query('ref'))
        {
            $status['valid'] = 1;
            
            $order_id = $request->query('ref');

            $transaction =  Transaction::where(function($query) use($order_id)
                            {
                              $query->where('order_id', $order_id)
                                    ->orWhere('transaction_id', $order_id)
                                    ->orWhere('reference_id', $order_id);
                            })
                            ->where(['processor' => $this->name, 'status' => 'pending'])
                            ->first() ?? abort(404);

            $response = $this->verify_payment($transaction);

            if($this->error_msg)
            { 
    	        	return $response;
            }

            if($transaction)
            {
                $transaction->status = $response['status'] === true ? 'paid' : 'pending';
                $transaction->transaction_id = $request->post('id');

                $transaction->save();

                $response['status'] = 1;
                $response['transaction'] = $transaction;
            }
        }

        return $response;
    }
  }