<?php

	namespace App\Libraries;

	use App\User;
	use Illuminate\Support\Facades\{ DB, Session, Cache, Validator, Auth };
  use Illuminate\Http\Request;
  use App\Models\{ Transaction };
  use GuzzleHttp\{ Client };


	class Razorpay
	{
		public $name = "razorpay";
		public $callback_url;
		public $cancel_url;
		public $supported_currencies = ["AED","ALL","AMD","ARS","AUD","AWG","BBD","BDT","BMD","BND","BOB","BSD","BWP","BZD","CAD","CHF","CNY","COP","CRC","CUP","CZK","DKK","DOP","DZD","EGP","ETB","EUR","FJD","GBP","GHS","GIP","GMD","GTQ","GYD","HKD","HNL","HRK","HTG","HUF","IDR","ILS","INR","JMD","KES","KGS","KHR","KYD","KZT","LAK","LBP","LKR","LRD","LSL","MAD","MDL","MKD","MMK","MNT","MOP","MUR","MVR","MWK","MXN","MYR","NAD","NGN","NIO","NOK","NPR","NZD","PEN","PGK","PHP","PKR","QAR","RUB","SAR","SCR","SEK","SGD","SLL","SOS","SSP","SVC","SZL","THB","TTD","TZS","USD","UYU","UZS","YER","ZAR"];
		public $currency_code;
		public $exchange_rate = 1;
		public $decimals;
		public $details = [];
		public $error_msg;
		public static $response = "default";


		public function __construct()
		{
				exists_or_abort(config("payments_gateways.{$this->name}"), __(":payment_proc is not enabled", ["payment_proc" =>  "Razorpay"]));

				prepare_currency($this);

				$this->details = [
						"items" => [],
						"total_amount" => 0,
						"currency" => $this->currency_code,
						"exchange_rate" => $this->exchange_rate,
						"custom_amount" => null,
						"reference" => generate_transaction_ref(),
						"transaction_id" => null,
		        "order_id" => null,
				];

				$key_id      = config("payments_gateways.{$this->name}.client_id");
				$key_secret  = config("payments_gateways.{$this->name}.secret_id");

				$this->token = base64_encode("{$key_id}:{$key_secret}");

				$this->callback_url = route('home.checkout.order_completed', ['processor' => $this->name, 'ref' => $this->details['reference']]);
		    $this->cancel_url   = route('home');
		}




		public function create_payment_link(float $amount)
		{			
				/*
					DOC : https://razorpay.com/docs/payment-links/api/v1/create/#request-parameters
					---------------------------------------------------------------------
					curl -u rzp_test_f8ONjSq19XZhLH:M6NbpABbYBmKz6noDAYOTTxZ \
					-X POST https://api.razorpay.com/v1/invoices/ \
					-H "Content-type: application/json" \
					-d "{
						"customer": {
							"name": "Acme Enterprises",
							"email": "admin@aenterprises.com",
							"contact": "9999999999"
						},
						"type": "link",
						"view_less": 1,
						"amount": 670042,
						"currency": "INR",
						"description": "Payment Link for this purpose - cvb.",
						"receipt": "TS91",
						"reminder_enable": true,
						"sms_notify": 1,
						"email_notify": 1,
						"expire_by": 1793630556,
						"callback_url": "https://example-callback-url.com/",
						"callback_method": "get"
					}"
				*/

				$this->details['total_amount'] = $amount;

				$amount = (int)ceil(format_amount($amount * $this->exchange_rate, false, $this->decimals) * pow(10, $this->decimals));

				$payload = [
					"type" => "link",
					"view_less" => 1,
					"amount" => $amount,
					"currency" => $this->currency_code,
					"reminder_enable" => false,
					"description" => __("Purchase from :app_name", ["app_name" => mb_ucfirst(config("app.name"))]),
					"sms_notify" => 0,
					"email_notify" => 0,
					"callback_url" => $this->callback_url,
					"callback_method" => "get"
				];

				$client = new Client([
	          "verify" => false,
	          "http_errors" => false,
	          "headers" => [
	          		"Authorization" => "Basic {$this->token}",
	              "Content-Type" => "application/json"
	          ]
	      ]);

	      $response = $client->request("POST", "https://api.razorpay.com/v1/invoices", ["json" => $payload]);

	      if(stripos($response->getStatusCode(), '20') === false)
	      {
	          $this->error_msg = ["user_message" => (string)$response->getBody()];

	          return;
	      }

	      $response = json_decode((string)$response->getBody(), true);

				if(isset($response['error']))
				{
						$this->error_msg = ["user_message" => $response['error']['description']];

						return;
				}

				return $response;
		}



		public function fetch_payment(string $razorpay_payment_id)
		{
				/*
					DOC : https://razorpay.com/docs/api/payments/#fetch-a-payment
					--------------------------------------------------------------
					curl -u rzp_test_f8ONjSq19XZhLH:M6NbpABbYBmKz6noDAYOTTxZ \
					-X GET https://api.razorpay.com/v1/payments/pay_ELaVWiGiOQMkOB
				*/

				$client = new Client([
	          "verify" => false,
	          "http_errors" => false,
	          "headers" => [
	          		"Authorization" => "Basic {$this->token}"
	          ]
	      ]);

	      $response = $client->request("GET", "https://api.razorpay.com/v1/payments/{$razorpay_payment_id}");

	      if(stripos($response->getStatusCode(), '20') === false)
	      {
	          $this->error_msg = ["user_message" => (string)$response->getBody()];

	          return;
	      }

	      $response = json_decode((string)$response->getBody(), true);

				if(isset($response['error']))
				{
						$this->error_msg = ["user_message" => $response['error']['description']];

						return;
				}

				return $response;
		}



		public function fetch_invoice(string $razorpay_invoice_id)
		{
				/*
					DOC : https://razorpay.com/docs/api/invoices/#fetch-an-invoice
					--------------------------------------------------------------
					curl -u rzp_test_32hsbEKriO6ai4:SC6d7z4FcgX2wJj49obRRT4M \
					https://api.razorpay.com/v1/invoices/inv_gHQwerty123ggd
				*/

				$client = new Client([
	          "verify" => false,
	          "http_errors" => false,
	          "headers" => [
	          		"Authorization" => "Basic {$this->token}"
	          ]
	      ]);

	      $response = $client->request("GET", "https://api.razorpay.com/v1/invoices/{$razorpay_invoice_id}");

	      if(stripos($response->getStatusCode(), '20') === false)
	      {
	          $this->error_msg = ["user_message" => (string)$response->getBody()];

	          return;
	      }

	      $response = json_decode((string)$response->getBody(), true);

				if(isset($response['error']))
				{
						$this->error_msg = ["user_message" => $response['error']['description']];

						return;
				}

				return $response;
		}



		public function refund_payment(string $payment_id, int $amount)
		{
				/*
					DOC : https://razorpay.com/docs/api/refunds/#refund-a-payment
					---------------------------------------------------------------
					curl -u <YOUR_KEY_ID>:<YOUR_KEY_SECRET> \
						-X POST \
						https://api.razorpay.com/v1/payments/pay_29QQoUBi66xm2f/refund
						-H "Content-Type: application/json" \
						-d "{
							"amount": 20000"
					}"
				*/

				$payload = [];

				if($amount)
				{
						$payload['amount'] = round($amount*100);
				}

				$client = new Client([
	          "verify" => false,
	          "http_errors" => false,
	          "headers" => [
	          		"Authorization" => "Basic {$this->token}",
	          		"Content-Length" => strlen(json_encode($payload)),
	          ]
	      ]);

	      $response = $client->request("POST", "https://api.razorpay.com/v1/payments/{$payment_id}/refund", ['json' => $payload]);

	      if(stripos($response->getStatusCode(), '20') === false)
	      {
	          $this->error_msg = ["user_message" => (string)$response->getBody()];

	          return;
	      }

	      $response = json_decode((string)$response->getBody(), true);

				if(isset($response['error']))
				{
						$this->error_msg = ["user_message" => $response['error']['description']];

						return;
				}

				return $response;
		}



		public function init_payment(float $amount)
		{
	      if($this->error_msg)
	      {
	        	return $this->error_msg;
	      }

	      $response = $this->create_payment_link($amount);

	      if($this->error_msg)
	      {
	        	return $this->error_msg;
	      }

	      if($response)
	      {
		      	$this->details['transaction_id'] = $response['order_id'];
		      	$this->details['order_id'] = $response['id'];

		        return urldecode($response['short_url']) ?? '/';
	      }
		}



		public function complete_payment(Request $request)
    {
	      if(stripos($request->processor, $this->name) !== false && $request->ref !== null)
	      {
		        $transaction_id = $request->ref;

		        $transaction =  Transaction::where(['processor' => $this->name])
		                        ->where(function($builder) use($transaction_id)
		                        {
			                          $builder->where(['transaction_id' => $transaction_id])
			                                  ->orWhere(['order_id' => $transaction_id])
			                                  ->orWhere(['reference_id' => $transaction_id]);
		                        })
		                        ->first();

		        if(!$transaction)
		        {
			          return [
			            'status' => false, 
			            'user_message' => __('Missing transaction database record [:transaction_id].', ['transaction_id' => $transaction_id])
			          ];
		        }

		        if($transaction->status !== 'paid')
		        {
			          $transaction->status = 'pending';
			          $transaction->save();
		        }

		        return ['status' => true, 'user_message' => null, 'transaction' => $transaction];
	      }

	      return ['status' => false, 'user_message' => __('Something wrong happened.')];
    }



		public function handle_webhook_notif(Request $request)
    {
	      $response = ['status' => 0, 'transaction' => null, 'valid' => 0];

	      if(stripos($request->processor, $this->name) !== false)
	      {        
		        $status['valid'] = 1;

		        $expected_sig   = $request->header('x-razorpay-signature');
		        $calculated_sig = hash_hmac('sha512', file_get_contents("php://input"), config("payments_gateways.{$this->name}.webhook_secret"));
		        $captured       = $request->input('payload.payment.entity.captured') === true;
		        $order_id 		  = $request->input('payload.payment.entity.order_id');

		        if($captured && ($calculated_sig === $expected_sig) && $order_id !== null)
		        {
			          $transaction =  Transaction::where(function($query) use($order_id)
			                          {
				                            $query->where('order_id', $order_id)
				                                  ->orWhere('transaction_id', $order_id)
				                                  ->orWhere('reference_id', $order_id);
			                          })
			                          ->where(['processor' => $this->name, 'status' => 'pending'])
			                          ->first();

			          if($transaction)
			          {
				            $transaction->status = 'paid';
				            $transaction->confirmed = 1;

				            $transaction->save();

				            $response['status'] = 1;
				            $response['transaction'] = $transaction;
			          }
		        }
	      }

	      return $response;
    }
	}