<?php

	namespace App\Libraries;

  use App\User;
  use Illuminate\Support\Facades\{ DB, Session, Cache, Validator, Auth };
	use Illuminate\Http\Request;
  use App\Models\{ Transaction };
  use GuzzleHttp\{ Client };

	class Paystack 
	{
		public $name = "paystack";
		public $return_url;
		public $cancel_url;
		public $supported_currencies = ["USD", "GHS", "NGN"];
		public $currency_code;
		public $exchange_rate = 1;
		public $decimals;
		public $details  = [];
		public $channels = ["card", "bank", "ussd", "qr", "mobile_money", "bank_transfer"];
		public $error_msg;
		public static $response = "default";


		public function __construct()
		{
				exists_or_abort(config("payments_gateways.{$this->name}.enabled"), __(":payment_proc is not enabled", ["payment_proc" =>  ucfirst($this->name)]));
	            
				$this->currency_code = config("payments.currency_code");
				$this->channels 		 = array_filter(explode(",", config("payments_gateways.{$this->name}.channels"))) ?? $this->channels;
				$this->decimals 		 = config("payments.currencies.{$this->currency_code}.decimals");

				prepare_currency($this);

	      $this->details = [
		      	"items" => [],
			      "total_amount" => 0,
			      "currency" => $this->currency_code,
			      "exchange_rate" => $this->exchange_rate,
			      "custom_amount" => null,
			      "reference" => generate_transaction_ref(),
			      "transaction_id" => null,
			      "order_id" => null,
		    ];

		    $this->return_url  = route("home.checkout.order_completed", ['processor' => $this->name, 'ref' => $this->details['reference']]);
				$this->cancel_url  = route('home');
		}


    
		public function create_transaction(float $amount, string $user_email)
		{
				$this->details['total_amount'] = $amount;

				$amount = (int)ceil(format_amount($amount * $this->exchange_rate, false, $this->decimals) * pow(10, $this->decimals));

				/* DOC : https://paystack.com/docs/api/#transaction-initialize
				--------------------------------------------------------------
					curl https://api.paystack.co/transaction/initialize
					-H "Authorization: Bearer YOUR_SECRET_KEY"
					-H "Content-Type: application/json"
					-d "{ email: "customer@email.com", amount: "20000" }"
					-X POST*/

				$payload = [
						"email" 			 => $user_email,
		        "amount" 			 => ceil($amount),
		        "currency" 		 => $this->currency_code,
		        "callback_url" => $this->return_url,
		        "channels" 		 => $this->channels,
		        "metadata" 		 => [
			        	"cancel_action" => $this->cancel_url, 
			        	"line_items" => [
				        		[
							        "name" => __("Purchase from :app_name", ["app_name" => mb_ucfirst(config("app.name"))]),
							        "amount" => $amount
							      ]
			        	]
		        ]
	      ];
	      
				$api_url = "https://api.paystack.co/transaction/initialize";

	      $secret_key  = config("payments_gateways.{$this->name}.secret_key");

	      $client = new Client([
		      	'verify' => false,
		      	'http_errors' => false,
		      	'headers' => [
		      			"Content-type" => "application/json", 
		      			"Authorization" => "Bearer {$secret_key}"
		      	]
	      ]);

	      $response = $client->request('POST', 'https://api.paystack.co/transaction/initialize', ['json' => $payload]);

	      if(stripos($response->getStatusCode(), '20') === false)
	      {
	      		$this->error_msg = ['user_message' => __('Request returned bad response code')];

	      		return;
	      }

	      $response = json_decode((string)$response->getBody(), true);

	      if(!$response['status'])
	      {
	      		$this->error_msg = ['user_message' => $response['message']];

	      		return;
	      }

				return $response;
		}



		public function verify_transaction($reference)
		{

			/* DOC : https://paystack.com/docs/api/#transaction-verify
			----------------------------------------------------------
			curl https://api.paystack.co/transaction/verify/:reference
			-H "Authorization: Bearer YOUR_SECRET_KEY"
			-X GET
			*/

			$ch 		 = curl_init();
			$api_url = "https://api.paystack.co/transaction/verify/{$reference}";

      $secret_key  = config("payments_gateways.{$this->name}.secret_key");


      curl_setopt($ch, CURLOPT_HTTPGET, 1);
			curl_setopt($ch, CURLOPT_URL, $api_url); 
			curl_setopt($ch, CURLOPT_HTTPHEADER, ["Content-type: application/json", "Authorization: Bearer {$secret_key}"]);
			curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);

			$result = curl_exec($ch);
			
			if(curl_errno($ch) || !json_decode($result))
			{
				$error_msg = curl_error($ch);

				curl_close($ch);

				$this->error_msg = ["user_message" => $error_msg];

				return;
	    }
	    
	    curl_close($ch);

	    $result = json_decode($result);

	    if(!$result->status)
	    {
	      $this->error_msg = ["user_message" => $result->message];

				return;
	    }

			return $result;
		}



		public function fetch_transaction($transaction_id)
		{
			/* DOC : https://paystack.com/docs/api/#transaction-fetch
			---------------------------------------------------------
			curl https://api.paystack.co/transaction/:id
			-H "Authorization: Bearer YOUR_SECRET_KEY"
			-X GET
			*/

			$ch 		 = curl_init();
			$api_url = "https://api.paystack.co/transaction/{$transaction_id}";

      $secret_key  = config("payments_gateways.{$this->name}.secret_key");

      $client = new Client([
	      	'verify' => false,
	      	'http_errors' => false,
	      	'headers' => [
	      			"Content-type" => "application/json", 
	      			"Authorization" => "Bearer {$secret_key}"
	      	]
      ]);

      $response = $client->request('GET', "https://api.paystack.co/transaction/{$transaction_id}", ['json' => $payload]);

      if(stripos($response->getStatusCode(), '20') === false)
      {
      		$this->error_msg = ['user_message' => __('Request returned bad response code')];

      		return;
      }

      $response = json_decode((string)$response->getBody(), true);

      if(!$response['status'])
      {
	        $this->error_msg = ["user_message" => $response['message']];

					return;
      }

      return $response;
		}



		public function init_payment(float $amount, string $user_email = null)
		{
				$request = request();

	      $user_email = $user_email ?? request()->user()->email ?? null;

	      if(!$user_email)
	      {
		        $request->validate(['email' => 'email|required']);

		        $user_email = $request->post('email');
	      }

	      $response = $this->create_transaction($amount, $user_email);
	      
	      if($this->error_msg)
	      {
	      		return $this->error_msg;
	      }

	      $this->details['transaction_id'] = $response['data']['reference'];
	      $this->details['order_id'] 			 = $response['data']['access_code'];

	      return $response['data']['authorization_url'];
		}



		public function complete_payment(Request $request)
    {
      if(stripos($request->processor, $this->name) !== false && $request->ref !== null)
      {
	        $transaction_id = $request->ref;

	        $transaction =  Transaction::where(['processor' => $this->name])
	                        ->where(function($builder) use($transaction_id)
	                        {
		                          $builder->where(['transaction_id' => $transaction_id])
		                                  ->orWhere(['order_id' => $transaction_id])
		                                  ->orWhere(['reference_id' => $transaction_id]);
	                        })
	                        ->first();

	        if(!$transaction)
	        {
		          return [
			            'status' => false, 
			            'user_message' => __('Missing transaction database record [:transaction_id].', ['transaction_id' => $transaction_id])
		          ];
	        }

	        if($transaction->status !== 'paid')
	        {
		          $transaction->status = 'pending';
		          $transaction->save();
	        }

	        return ['status' => true, 'user_message' => null, 'transaction' => $transaction];
      }

      return ['status' => false, 'user_message' => __('Something wrong happened.')];
    }



		public function handle_webhook_notif(Request $request)
    {
    	$notif    = json_decode(file_get_contents("php://input"));
      $response = ['status' => 0, 'transaction' => null, 'valid' => 0];
      $event    = $notif->event ?? null;

      if(stripos($request->processor, $this->name) !== false && $request->input('event') === 'charge.success' && $request->input('data.reference'))
      {        
	        $status['valid'] = 1;

	        $expected_sig   = $request->header('X-Paystack-Signature');
	        $calculated_sig = hash_hmac('sha512', file_get_contents("php://input"), config("payments_gateways.{$this->name}.secret_key"));
	        $success        = strtolower($request->input('data.status')) === 'success';

	        if($success && ($calculated_sig === $expected_sig))
	        {
		          $order_id = $request->input('data.reference');

		          $transaction =  Transaction::where(function($query) use($order_id)
		                          {
			                            $query->where('order_id', $order_id)
			                                  ->orWhere('transaction_id', $order_id)
			                                  ->orWhere('reference_id', $order_id);
		                          })
		                          ->where(['processor' => $this->name, 'status' => 'pending'])
		                          ->first();

		          if($transaction)
		          {
			            $transaction->status = 'paid';
			            $transaction->confirmed = 1;

			            $transaction->save();

			            $response['status'] = 1;
			            $response['transaction'] = $transaction;
		          }
	        }
      }

      return $response;
    }
	}