<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;


class Product extends Model
{    
    protected $guarded = [];



    public function sales()
    {
        return $this->hasMany(Transaction::class, 'product_id');
    }


    public function category()
    {
        return $this->belongsTo(Category::class);
    }


    public function keys($remaining = false)
    {
        $result = $this->hasMany(Key::class, 'product_id');

        if($remaining)
        {
            $result = $result->whereRaw('key_s.user_id IS NULL');
        }  
        
        return $result->get();
    }

    public function isPurchased()
    {
        if(\Auth::check())
        {
            return $this->sales->where('user_id', user('id'))->count() > 0;
        }

        return null;
    }


    public function isReviewed()
    {
        if(\Auth::check())
        {
            return $this->reviews->where('user_id', user('id'))->count() == 1;
        }

        return null;
    }


    public function reviews()
    {
        return $this->hasMany(Review::class)->orderBy('id', 'desc');
    }



    public function preview_is($name):bool
    {
        return $this->preview_type == $name;
    }



    public function preview_matches($regexp):bool
    {
        return preg_match($regexp, $this->preview_type);
    }


    public function has_preview(string $type = null):bool
    {
        return $type ? (!empty($this->preview) && $this->preview_is($type)) : !empty($this->preview); 
    }


    public function has_keys()
    {
        return \App\Models\Key::selectRaw("IF(COUNT(key_s.id) = 0, NULL, COUNT(key_s.id)) as count")
                                ->where('product_id', $this->id)->pluck('count')->first();
    }


    public function has_promo(): bool
    {
        if(isset($this->promo_price))
        {
            if(!json_decode($this->promo_time ?? '', true))
            {
                return true;
            }

            $promo_time = json_decode($this->promo_time ?? '', true);

            $from = \Carbon\Carbon::parse($promo_time['from'])->timestamp;
            $to   = \Carbon\Carbon::parse($promo_time['to'])->timestamp;

            return time() >= (int)$from && (int)$to >= time();
        }

        return false;
    }

    public function comments()
    {
        return $this->hasMany(Comment::class, 'product_id');
    }
}
